package com.atlassian.plugins.domain.wrapper;

import com.atlassian.plugins.domain.model.Link;
import com.atlassian.plugins.domain.model.category.Category;
import com.atlassian.plugins.domain.model.country.Country;
import com.atlassian.plugins.domain.model.plugin.*;
import com.atlassian.plugins.domain.model.product.CompatibilityMatrix;
import com.atlassian.plugins.domain.model.product.CompatibilityMatrixData;
import com.atlassian.plugins.domain.model.product.Product;
import com.atlassian.plugins.domain.model.product.ProductCompatibility;
import com.atlassian.plugins.domain.model.review.Review;
import com.atlassian.plugins.domain.model.user.Avatar;
import com.atlassian.plugins.domain.model.user.User;
import com.atlassian.plugins.domain.model.vendor.SupportDetail;
import com.atlassian.plugins.domain.model.vendor.SupportOrganisation;
import com.atlassian.plugins.domain.model.vendor.Vendor;
import com.atlassian.plugins.domain.model.vendor.VendorRelationship;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import java.util.*;

/**
 * Why this class was initially created :
 *
 *  Due to some Jersey + JAXB problems, we need to wrap lists of items so that they can be marshalled and unmarshalled
 *  properly.
 *
 *  This class wraps a list of items, causing the root element to be "items" and each element inside to be "item",
 *  with each item element containing a type attribute which points to the object type.
 *
 *  The error returned by Jersey, that this class avoids, is:
 *    A message body writer for Java type, class java.util.ArrayList, and MIME media type, text/xml, was not found
 *
 * Why this class was extended :
 *
 *  To support the dual-expansions of lists, we need to be able to control what attributes are rendered on a list,
 *  hence this class expand and size attributes. These expand elements are only used when the ListWrapper bring rendered
 *  belongs to another object (ie not when a retrieve method is called).
 *
 * Implemented the Iterable interface for client convenience - ie so it can be used in a foreach loop. I would also
 * like to implement List<T> but that completely stuffs up the JAXB rendering.
 *
 */
@XmlSeeAlso({Link.class,
    Category.class, Country.class,
    Contributor.class, Download.class, Icon.class, Requirement.class, Plugin.class, PluginVersion.class, Screenshot.class,
    Product.class, ProductCompatibility.class, CompatibilityMatrix.class, CompatibilityMatrixData.class,
    Review.class,
    Avatar.class, User.class,
    PluginEntry.class,
    SupportDetail.class, SupportOrganisation.class, Vendor.class, VendorRelationship.class, PluginCompatibilityStatus.class
    })
@XmlRootElement(name="items")
public class ListWrapper<T> implements Iterable<T> {

    private String expand;

    private List<T> list;

    private Integer setSize;

    // -- constructors -- //

    public ListWrapper() {
        this(null);
    }

    public ListWrapper(List<T> list) {
        if (list == null) {
            this.list = new ArrayList<T>();
        } else {
            this.list = list;
        }
    }

    // -- jaxb annotated methods -- //

    /*
     * JAXB foo required for proper rendering of size attribute.
     * Must only count non-null elements.
     */
    @XmlAttribute(name="size")
    private int getSize_() {
        if (setSize != null) return setSize;
        return list.size();
    }

    @XmlAttribute
    public String getExpand() {
        return expand;
    }

    public void setExpand(String expand) {
        this.expand = expand;
    }

    @XmlElement(name="item")
    public List<T> getList() {
        return list;
    }

    public void setList(List<T> list) {
        this.list = list;
    }

    /**
     * Wraps a new, empty list
     */
    public void reset() {
        list = new ArrayList<T>();
    }

    /**
     * Records the number of non-null elements
     */
    public void rememberSize() {
        int total = 0;
        for (Object obj : list) {
            if (obj != null) total++;
        }
        this.setSize = total;
    }

    // -- wrapped list methods -- //

    public boolean add(T t) {
        return list.add(t);
    }

    public boolean remove(Object o) {
        return list.remove(o);
    }

    public boolean containsAll(Collection<?> objects) {
        return list.containsAll(objects);
    }

    public boolean addAll(Collection<? extends T> ts) {
        return list.addAll(ts);
    }

    public boolean addAll(int i, Collection<? extends T> ts) {
        return list.addAll(i, ts);
    }

    public boolean removeAll(Collection<?> objects) {
        return list.removeAll(objects);
    }

    public boolean retainAll(Collection<?> objects) {
        return list.retainAll(objects);
    }

    public void clear() {
        list.clear();
    }

    public T set(int index, T t) {
        return list.set(index, t);
    }

    public void add(int i, T t) {
        list.add(i, t);
    }

    public T remove(int i) {
        return list.remove(i);
    }

    public int indexOf(Object o) {
        return list.indexOf(o);
    }

    public int lastIndexOf(Object o) {
        return list.lastIndexOf(o);
    }

    public ListIterator<T> listIterator() {
        return list.listIterator();
    }

    public ListIterator<T> listIterator(int i) {
        return list.listIterator(i);
    }

    public List<T> subList(int i, int i1) {
        return list.subList(i, i1);
    }

    public T get(int index) {
        return list.get(index);
    }

    public int size() {
        return list.size();
    }

    public boolean isEmpty() {
        return list.isEmpty();
    }

    public boolean contains(Object o) {
        return list.contains(o);
    }

    public Iterator<T> iterator() {
        return list.iterator();
    }

    public Object[] toArray() {
        return list.toArray();
    }

    public <T> T[] toArray(T[] ts) {
        return list.toArray(ts);
    }


}
