package com.atlassian.plugins.client.service;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;

import com.atlassian.plugins.PacException;
import com.atlassian.plugins.domain.DTO;
import com.atlassian.plugins.domain.authorisation.AuthorisationException;
import com.atlassian.plugins.domain.model.Status;
import com.atlassian.plugins.domain.search.SearchCriteria;
import com.atlassian.plugins.domain.wrapper.ListWrapper;
import com.atlassian.plugins.service.RestService;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientRequest;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.client.filter.ClientFilter;
import com.sun.jersey.client.apache.ApacheHttpClient;
import com.sun.jersey.client.apache.ApacheHttpClientHandler;
import com.sun.jersey.core.util.MultivaluedMapImpl;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;

import static org.apache.commons.codec.binary.Base64.encodeBase64;

public abstract class AbstractRestServiceClient<T extends DTO> implements RestService<T> {

    public static final String VERSION = "1.0";
    public static final String PROXY_AUTHORIZATION = "Proxy-Authorization";

    /*
     * The class that will be returned by queries to the particular service
     */
    protected abstract Class<T> getEntity();

    /*
     * The path for all queries to the particular service
     */
    protected abstract String getPath();

    /**
     * The base URL for all queries to the service.
     */
    private String baseUrl;

    public void setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
    }

    /**
     * If supplied with a password, the username used to authenticate with the server
     */
    private String username;

    public void setUsername(String username) {
        this.username = username;
    }

    public String getUsername() {
        return username;
    }

    /**
     * If supplied with a username, the password used to authenticate with the server
     */
    private String password;

    public void setPassword(String password) {
        this.password = password;
    }

    public String getPassword() {
        return password;
    }

    /**
     * If supplied with an alternativeUsername, the user to act as when authenticating as an admin
     */
    private String alternativeUsername;

    public void setAlternativeUsername(String alternativeUsername) {
        this.alternativeUsername = alternativeUsername;
    }

    public String getAlternativeUsername() {
        return alternativeUsername;
    }

    /**
     * Whether logging should be turned on
     */
    private boolean logWebResourceRequests = false;

    public void setLogWebResourceRequests(boolean logWebResourceRequests) {
        this.logWebResourceRequests = logWebResourceRequests;
    }

    protected WebResource getWebResource() {
        ClientConfig config = new DefaultClientConfig();
        HttpClient httpClient = new HttpClient(new MultiThreadedHttpConnectionManager());
        String host = System.getProperty("http.proxyHost");
        Integer port = Integer.getInteger("http.proxyPort");
        if (host != null && port != null)
        {
            httpClient.getHostConfiguration().setProxy(host, port);
        }

        ApacheHttpClientHandler handler = new ApacheHttpClientHandler(httpClient);
        Client client = new ApacheHttpClient(handler, config);
        addProxyAuthFilter(client);

        WebResource resource = client.resource(baseUrl + "/" + VERSION + getPath() + "/");
        if (logWebResourceRequests) resource.addFilter(new com.sun.jersey.api.client.filter.LoggingFilter());
        return resource;
    }

    protected WebResource.Builder buildWebResource(WebResource resource) {

        String encoding = null;
        if (username != null && password != null) {
            String userPassword;
            if (alternativeUsername == null) {
                userPassword = username + ":" + password;
            } else {
                userPassword = username + " " + alternativeUsername + ":" + password;
            }
            encoding = new sun.misc.BASE64Encoder().encode(userPassword.getBytes());

            //the encoding may be fairly long so remove any stupid newlines that appear
            //refer http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6459815
            //should be fixed for Java 7 ... awesome
            encoding = encoding.replaceAll("\r", "").replaceAll("\n", "");
        }

        //no encoding, do not include the header
        if (encoding == null) return resource.type(CONTENT_XML);

        return resource.type(CONTENT_XML).header("Authorization", "Basic " + encoding);
    }

    @SuppressWarnings("unchecked")
    protected Object getEntityFromResponse(ClientResponse response, Class entityClass) {

        //check that response was not forbidden
        try {
            if (response.getStatus() == Response.Status.FORBIDDEN.getStatusCode())
                throw new AuthorisationException(response.getEntity(Status.class));
        } catch (AuthorisationException e) {
            throw e;
        } catch (Exception e) {
            throw new AuthorisationException(String.valueOf(Response.Status.FORBIDDEN.getStatusCode()), e);
        }

        //check that the response was not a 500
        try {
            if (response.getStatus() == Response.Status.INTERNAL_SERVER_ERROR.getStatusCode()) {
                //sometimes 401s are thrown as 500s ... not sure why
                Status status = response.getEntity(Status.class);
                if (Response.Status.FORBIDDEN.toString().equals(status.getCode())) {
                    throw new AuthorisationException(status);
                } else {
                    throw new PacException(status);
                }
            }
        } catch (AuthorisationException e) {
            throw e;
        } catch (PacException e) {
            throw e;
        } catch (Exception e) {
            throw new PacException(String.valueOf(Response.Status.INTERNAL_SERVER_ERROR.getStatusCode()), e);
        }

        //try to get the entity from the response
        if (entityClass == null) return null;

        try {
            return response.getEntity(entityClass);

        } catch (Exception e) {

            final Status entity;
            try {
                //on failure, there might be a Status entity, so try and get that
                entity = response.getEntity(Status.class);

            } catch (Exception ex) {
                //otherwise, just use a generic code
                throw new PacException(String.valueOf(response.getStatus()), e);
            }

            throw new PacException(entity, e);
        }
    }

    public String create(T obj) {

        // Objects being sent back to the server needs to be expanded so that all the fields are encoded in the XML
        obj.setExpanded(true);

        ClientResponse response = buildWebResource(getWebResource()).post(ClientResponse.class, obj);

        //this call will handle 401s and 500s
        Status status = (Status) getEntityFromResponse(response, Status.class);

        if (response.getStatus() == Response.Status.CREATED.getStatusCode()) {
            return status.getSubCode();
        }

        throw new PacException("Unexpected Response: " + response.getStatus(), status);
    }

    public T retrieve(String id) {
        return retrieve(id, null);
    }

    @SuppressWarnings("unchecked")
    public T retrieve(String id, List<String> expand) {
        WebResource webResource = getWebResource();

        if (expand != null && !expand.isEmpty()) {
            MultivaluedMapImpl map = new MultivaluedMapImpl();

            StringBuffer sb = new StringBuffer();
            for (String ex : expand) {
                if (sb.length() > 0) sb.append(PARAM_VALUE_SEPARATOR);
                sb.append(ex);
            }
            map.putSingle(PARAM_EXPAND, sb.toString());

            webResource = webResource.queryParams(map);
        }
        
        webResource = webResource.path(id);

        ClientResponse cr = buildWebResource(webResource).get(ClientResponse.class);

        if (cr.getStatus() == Response.Status.NOT_FOUND.getStatusCode()) return null;

        //this call will handle 401s and 500s
        return (T) getEntityFromResponse(cr, getEntity());
    }

    public String update(String id, T obj) {
        WebResource webResource = getWebResource();
        webResource = webResource.path(id);

        // Objects being sent back to the server needs to be expanded so that all the fields are encoded in the XML
        obj.setExpanded(true);

        ClientResponse response = buildWebResource(webResource).put(ClientResponse.class, obj);

        //this call will handle 401s and 500s
        Status status = (Status) getEntityFromResponse(response, Status.class);

        if (response.getStatus() == Response.Status.OK.getStatusCode()) {
            return status.getSubCode();
        }

        throw new PacException("Unexpected Response: " + response.getStatus(), status);
    }

    public void delete(String id) {
        WebResource webResource = getWebResource();
        webResource = webResource.path(id);
        ClientResponse response = buildWebResource(webResource).delete(ClientResponse.class);
        //this call will handle 401s and 500s
        getEntityFromResponse(response, null);
    }

    @SuppressWarnings("unchecked")
    public List<T> list(Integer max, Integer offset, List<String> expand) {

        //assemble the query
        WebResource webResource = getWebResource();

        MultivaluedMapImpl map = new MultivaluedMapImpl();
        if (max != null) map.putSingle(PARAM_MAX, max);
        if (offset != null) map.putSingle(PARAM_OFFSET, offset);

        if (expand != null && !expand.isEmpty()) {

            StringBuffer sb = new StringBuffer();
            for (String ex : expand) {
                if (sb.length() > 0) sb.append(PARAM_VALUE_SEPARATOR);
                sb.append(ex);
            }
            map.putSingle(PARAM_EXPAND, sb.toString());

        }

        webResource = webResource.queryParams(map);

        //make the call and convert to a proper list
        ClientResponse response = buildWebResource(webResource).get(ClientResponse.class);

        //this call will handle 401s and 500s
        ListWrapper listWrapper = (ListWrapper) getEntityFromResponse(response, ListWrapper.class);
        if (listWrapper.getList() == null) return new ArrayList<T>();
        return listWrapper.getList();
    }

    @SuppressWarnings("unchecked")
    public List<T> find(SearchCriteria criteria, List<String> expand) {

        WebResource webResource = getWebResource();

        MultivaluedMapImpl map = new MultivaluedMapImpl();

        if (expand != null && !expand.isEmpty()) {

            StringBuffer sb = new StringBuffer();
            for (String ex : expand) {
                if (sb.length() > 0) sb.append(PARAM_VALUE_SEPARATOR);
                sb.append(ex);
            }
            map.putSingle(PARAM_EXPAND, sb.toString());

        }

        webResource = webResource.queryParams(map);

        webResource = webResource.path(PATH_FIND);

        //make the call and convert to a proper list
        ClientResponse response = buildWebResource(webResource).type(CONTENT_XML).post(ClientResponse.class, criteria);

        //this call will handle 401s and 500s
        ListWrapper listWrapper = (ListWrapper) getEntityFromResponse(response, ListWrapper.class);
        if (listWrapper.getList() == null) return new ArrayList<T>();
        return listWrapper.getList();
    }

    public Long count(SearchCriteria criteria) {
        WebResource webResource = getWebResource();
        webResource = webResource.path(PATH_COUNT);

        ClientResponse response = buildWebResource(webResource).type(CONTENT_XML).post(ClientResponse.class, criteria);

        //this call will handle 401s and 500s
        String count = (String) getEntityFromResponse(response, String.class);        
        return new Long(count);
    }

    private void addProxyAuthFilter(Client client)
    {
        String proxyUser = System.getProperty("http.proxyUser");
        String proxyPassword = System.getProperty("http.proxyPassword");
        if (proxyUser != null && proxyPassword != null)
        {
            client.addFilter(new BasicProxyAuthFilter(proxyUser, proxyPassword));
        }
    }

    final class BasicProxyAuthFilter extends ClientFilter
    {
        private final String auth;

        BasicProxyAuthFilter(String username, String password)
        {
            try
            {
                auth = "Basic " + new String(encodeBase64((username + ":" + password).getBytes("ASCII")));
            }
            catch (UnsupportedEncodingException e)
            {
                throw new RuntimeException("That's some funky JVM you've got there", e);
            }
        }

        @Override
        public ClientResponse handle(ClientRequest cr) throws ClientHandlerException
        {
            MultivaluedMap<String, Object> headers = cr.getMetadata();
            if (!headers.containsKey(PROXY_AUTHORIZATION))
            {
                headers.add(PROXY_AUTHORIZATION, auth);
            }
            return getNext().handle(cr);
        }
    }

}
