package com.atlassian.plugins.less;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.lesscss.spi.DimensionAwareUriResolver;
import com.atlassian.lesscss.spi.UriResolver;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.elements.ResourceDescriptor;
import com.atlassian.plugin.webresource.transformer.TransformerParameters;
import com.atlassian.plugin.webresource.transformer.UrlReadingWebResourceTransformer;
import com.atlassian.webresource.api.prebake.DimensionAwareTransformerUrlBuilder;
import com.atlassian.webresource.api.prebake.DimensionAwareWebResourceTransformerFactory;
import com.atlassian.webresource.api.prebake.Dimensions;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

public class LessWebResourceTransformerFactory implements DimensionAwareWebResourceTransformerFactory {

    private final LessCompilerSupplier lessCompiler;
    private final PluginAccessor pluginAccessor;
    private final UriStateManager uriStateManager;
    private final UriResolverManager uriResolverManager;
    private final EventPublisher eventPublisher;

    public LessWebResourceTransformerFactory(LessCompilerSupplier lessCompiler,
                                             PluginAccessor pluginAccessor,
                                             UriStateManager uriStateManager,
                                             UriResolverManager uriResolverManager,
                                             EventPublisher eventPublisher) {
        this.lessCompiler = lessCompiler;
        this.pluginAccessor = pluginAccessor;
        this.uriStateManager = uriStateManager;
        this.uriResolverManager = uriResolverManager;
        this.eventPublisher = eventPublisher;
    }

    @Override
    public Dimensions computeDimensions() {
        Dimensions dimensions = Dimensions.empty();
        for (UriResolver uriResolver : uriResolverManager.getResolvers()) {
            if (uriResolver instanceof DimensionAwareUriResolver) {
                DimensionAwareUriResolver dimensionAwareUriResolver = (DimensionAwareUriResolver) uriResolver;
                dimensions = dimensions.product(dimensionAwareUriResolver.computeDimensions());
            }
        }
        return dimensions;
    }

    @Override
    public DimensionAwareTransformerUrlBuilder makeUrlBuilder(TransformerParameters params) {
        List<URI> resources = new ArrayList<>();
        if (params.isAmdModule()) {
            resources.add(UriUtils.resolvePluginUri(params.getPluginKey(), params.getAmdModuleLocation()));
        } else {
            String webResourceKey = params.getPluginKey() + ":" + params.getModuleKey();
            ModuleDescriptor descriptor = pluginAccessor.getEnabledPluginModule(webResourceKey);
            for (ResourceDescriptor resourceDescriptor : descriptor.getResourceDescriptors()) {
                if (resourceDescriptor.getLocation().endsWith(".less")) {
                    resources.add(UriUtils.resolveUri(
                        descriptor.getPluginKey(),
                        resourceDescriptor.getResourceLocationForName(null)
                    ));
                }
            }
        }
        return new LessTransformerUrlBuilder(resources, uriStateManager);
    }

    @Override
    public UrlReadingWebResourceTransformer makeResourceTransformer(TransformerParameters params) {
        return new LessWebResourceTransformer(lessCompiler, uriResolverManager, params.getPluginKey(), eventPublisher);
    }
}