package com.atlassian.plugins.less;

import java.net.URI;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.lesscss.LessCompiler;
import com.atlassian.lesscss.Loader;
import com.atlassian.lesscss.PluggableLoader;
import com.atlassian.lesscss.spi.UriResolver;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.QueryParams;
import com.atlassian.plugin.webresource.transformer.TransformableResource;
import com.atlassian.plugin.webresource.transformer.UrlReadingWebResourceTransformer;

import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.collect.Iterables;

/**
 * Transforms .less files into .css files, see http://lesscss.org/
 *
 * @since 1.0
 */
public class LessWebResourceTransformer implements UrlReadingWebResourceTransformer {

    /**
     * This is too expensive (memory and time) to construct every time. It is immutable, so just use a global one.
     */
    private final Supplier<LessCompiler> lessc;
    private final Loader loader;
    private final String pluginKey;
    private final UriResolverManager uriResolverManager;
    private final EventPublisher eventPublisher;

    public LessWebResourceTransformer(LessCompilerSupplier lessc, UriResolverManager uriResolverManager, String pluginKey, EventPublisher eventPublisher) {
        this.lessc = lessc;
        this.pluginKey = pluginKey;
        this.uriResolverManager = uriResolverManager;
        this.eventPublisher = eventPublisher;
        this.loader = new PluggableLoader(this.uriResolverManager.getResolvers());
    }

    @Override
    public DownloadableResource transform(TransformableResource transformableResource, QueryParams params) {
        final URI uri = UriUtils.resolveUri(pluginKey, transformableResource.location());

        UriResolver uriResolver = Iterables.find(uriResolverManager.getResolvers(), new Predicate<UriResolver>() {
            @Override
            public boolean apply(UriResolver resolver) {
                return resolver.supports(uri);
            }
        });

        URI preCompiledUri = PreCompilationUtils.resolvePreCompiledUri(uriResolver, uri);

        return preCompiledUri == null ?
                new LessResource(transformableResource.nextResource(), lessc.get(), loader, uri, eventPublisher) :
                new PreCompiledLessResource(transformableResource.nextResource(), uriResolver, preCompiledUri);
    }

}
