(function ($, _) {
    "use strict";

    AJS.namespace("JIRA.AttachImagesPlugin");

    var isIE8,
        isIE9,
        isIE10,
        appletNode;

    /**
     * Create a rejected jQuery deferred. All arguments are passed to .reject().
     * @returns a jQuery deferred.
     */
    function fail() {
        var deferred = $.Deferred();
        return deferred.reject.apply(deferred, arguments);
    }

    // It's a sad state of affairs that we need this browser detection, but unfortunately there's no reliable feature
    // detection for image pasting from clipboard support, so we're left with browser detection.
    //
    // JIRA politely puts classes on <html> to identify the browser for us, so let's treat that as our browser detection
    // API. We only care about IE for special casing clipboard behaviour, for all other browsers we'll assume they
    // support the HTML5 clipboard API,
    (function () {
        var classes = document.documentElement.className.split(/\s+/),
            isIE = $.inArray("msie", classes) > -1,
            gt7 = $.inArray("msie-gt-7", classes) > -1,
            gt8 = $.inArray("msie-gt-8", classes) > -1,
            gt9 = $.inArray("msie-gt-9", classes) > -1,
            gt10 = $.inArray("msie-gt-10", classes) > -1;

        isIE8  = isIE && gt7 && !gt8;
        isIE9  = isIE && gt8 && !gt9;
        isIE10 = isIE && gt9 && !gt10;
    })();

    JIRA.AttachImagesPlugin.polyfill = new (function () {
        var $node;

        // paste polyfill
        var handleKeydown = _.bind(function (event) {
            if (JIRA.AttachImagesPlugin.isPasteEvent(event)) {
                appletNode.reload();
                var imageUri = appletNode.getClipboardData();
                if (imageUri) {
                    var previewImageUri = appletNode.getClipboardData(this.pasteImageWidth, this.pasteImageHeight);
                    if (previewImageUri.length > this.dataURILimit) {
                        var f = this.dataURILimit / previewImageUri.length;
                        previewImageUri = appletNode.getClipboardData(this.pasteImageWidth * f, this.pasteImageHeight * f);
                    }
                    $node.trigger('polyPaste', [imageUri, previewImageUri]);
                }
            }
        }, this);

        /**
         * Install the polyfill to provide clipboard access to non-HTML5 browsers.
         *
         * @param {Element} node Where the 'keydown' event handler should be added.
         * @returns a jQuery promise that's resolved with [deployJava, contentWindow], or rejected
         *   with a [reason, message].
         *
         * Reasons include:
         * - "java-absent" -- Java isn't installed.
         * - "java-version" -- The version Java is not supported.
         * - "java-security" -- Java security settings are blocking the applet. Typically this can be overcome by
         *   lowering the 'Security Level' in Java's 'Control Panel' (e.g. from High to Medium).
         * - "unknown" -- Every other case.
         */
        this.install = _.once(function (node) {
            $node = $(node);

            return JIRA.AttachImagesPlugin.requireDeployJava()
                .pipe(function (deployJava, contentWindow) {
                    // Unfortunately using deployJava.versionCheck() isn't guaranteed to be accurate. Depending on the
                    // platform and JRE. For example Java 1.7.0_06 on IE9 only reports the JRE as 1.7.0 (i.e. no update
                    // information).
                    // Given that the specific rules/cases/etc are ambiguous, we'll opt for doing a 'best effort' here
                    // rather than hard coding a bunch of special case rules that aren't guaranteed to be reliable.
                    var attributes,
                        parameters,
                        requiresLegacyApplet,
                        minimumVersion = '1.7.0_06';

                    if (deployJava.getJREs().length === 0) {
                        return fail("java-absent");
                    } else if (!deployJava.versionCheck(minimumVersion + '+')) {
                        return fail("java-version");
                    }

                    // Java 1.7.0_45 changed the manifest attributes that are required to allow JavaScript to invoke
                    // applet methods. Java 1.7.0_40 and earlier requires the 'Trusted-Library' attribute, but later
                    // versions require the 'Caller-Allowable-Codebase' attribute.
                    //
                    // See https://blogs.oracle.com/java-platform-group/entry/7u45_caller_allowable_codebase_and for more
                    // details.
                    requiresLegacyApplet = !deployJava.versionCheck('1.7.0_45+');

                    attributes = {
                        id: 'JIRA HTML5 Images Applet',
                        codebase: AJS.contextPath() + "/download/resources/com.atlassian.plugins.jira-html5-attach-images:jira-html5-attach-images-resources/applet/",
                        code: "com.atlassian.plugins.jira.screenshot.applet.ScreenshotApplet.class",
                        archive: requiresLegacyApplet ? "screenshot-legacy.jar" : "screenshot.jar",
                        width: 0,
                        height: 0
                    };
                    parameters = {
                        permissions: "all-permissions"
                    };

                    deployJava.runApplet(attributes, parameters, minimumVersion);
                    appletNode = contentWindow.document.getElementById(attributes.id);

                    try {
                        // We need a try/catch here because...
                        // A 'appletNode.isSecurity' doesn't work, because it's falsey on IE. We can use .hasOwnProperty
                        // but it doesn't tell us if it's a function, and 'typeof appletNode.isSecurity' returns
                        // "unknown".
                        if (!appletNode || !appletNode.isSecurityOk()) {
                            return fail("java-security");
                        }
                    } catch (e) {
                        return fail("java-security");
                    }

                    $node.on('keydown', handleKeydown);
                    return executeAjaxUpload;
                }, function () {
                    return fail("unknown");
                })
                // Add a message to the error.
                .pipe(null, function (reason) {
                    var message;
                    if (reason === "unknown") {
                        message = AJS.I18n.getText("attach.screenshot.java.is.broken");
                    } else if (reason === "java-version") {
                        message = AJS.I18n.getText("attach.screenshot.java.is.old", "<a href=\"//java.com\">", "</a>");
                    } else if (reason === "java-absent") {
                        message = AJS.I18n.getText("attach.screenshot.java.is.absent", "<a href=\"//java.com\">", "</a>");
                    } else if (reason === "java-security") {
                        message = AJS.I18n.getText("attach.screenshot.java.is.blocked.by.security", "<a href=\"//java.com\">", "</a>");
                    }
                    return fail(reason, message);
                });
        });

        this.isRequired = function () {
            // Enable the Java applet for IE 8/9/10, assume all other browsers are compatible, since we
            // support latest IE, Chrome, Firefox, and all of these support HTML5 clipboard.
            return isIE8 || isIE9 || isIE10;
        };

        this.isRequiredForBinaryAjax = function () {
            // The applet has security restrictions on Windows 8 + IE10, so we can't use it there. Luckily IE10 supports
            // AJAX requests with binary data anyway, so we can use that. IE8 and IE9 don't support binary data.
            return isIE8 || isIE9;
        };

        this.pasteImageWidth = 510;
        this.pasteImageHeight = 280;
        this.dataURILimit = isIE8 ? 32000 : Number.MAX_VALUE;

        /**
         * Use the Java applet to make a HTTP request.
         * @param data
         * @param requestUrl
         * @returns a jQuery deferred that pretends to be a jqXHR. The only addition is a '.abort()' method to try to
         *   conform closer to jqXHR API.
         */
        function executeAjaxUpload(data, requestUrl) {
            var cookies = document.cookies,
                deferred = $.Deferred();

            appletNode.doMultipartRequest(requestUrl, "UTF-8", window.navigator.userAgent, deferred);
            // Java applets use the browser's cookie storage for URLConnection for non-http-only cookies. This means
            // xsrftoken and other cookies can pollute the browser's.
            deferred.always(function () {
                document.cookies = cookies;
            });
            return deferred;
        }

        // proxy ajax request, because session with temporary attachments is in applet
        this.proxyAjaxRequest = function (fn, userTokenFn) {
            return function () {
                var _smartAjaxMakeRequest = JIRA.SmartAjax.makeRequest;

                JIRA.SmartAjax.makeRequest = function (requestOptions) {
                    var requestParams = _.reduce(_.keys(requestOptions.data), function (r, name) {
                        r.push(name, requestOptions.data[name]);
                        return r;
                    }, []);

                    requestParams.push("secureToken", userTokenFn());

                    var requestResult = appletNode.doAjaxRequest(requestOptions.url, "UTF-8", window.navigator.userAgent, $.Deferred(), requestParams);
                    requestResult.abort = $.noop;
                    requestResult.then(function (data) {
                        var responseHeaders = _.filter(arguments, function (el, idx) {
                            return idx > 0;
                        });
                        responseHeaders[0] = "Status";
                        responseHeaders = _.reduce(_.reduce(responseHeaders, function (r, el, idx) {
                            var pdx = (idx / 2) << 0;
                            (r[pdx] || (r[pdx] = [])).push(el);
                            return r;
                        }, []), function (r, el) {
                            r[el[0]] = el[1];
                            return r;
                        }, {});

                        var xhr = {
                            getResponseHeader: function (name) {
                                return responseHeaders[name];
                            }
                        };
                        var textStatus = 'success';
                        var smartAjaxResult = new JIRA.SmartAjax.SmartAjaxResult(xhr, 0, "success", data, true);
                        requestOptions.complete(xhr, textStatus, smartAjaxResult);
                    });

                    return requestResult;
                };

                fn.apply(this, arguments);

                JIRA.SmartAjax.makeRequest = _smartAjaxMakeRequest;
            };
        };
    });
})(AJS.$, _);