package com.atlassian.plugins.avatar;

import com.atlassian.util.concurrent.Supplier;

/**
 * Wrapping representation of an application-specific domain object that can have an Avatar. Typical example is the user
 * object. Implementers of {@AvatarProvider} wishing to support {@link AvatarProvider#getAvatar(AvatarOwner, String)} to
 * enable domain object interrogation inside the Avatar retrieval.
 */
public interface AvatarOwner<T> extends Supplier<T> {

    /**
     * For users this is the email address. The implementation may support multiple unique identifiers for each domain
     * object but the value returned here must be capable of being used by the AvatarProvider for getting the avatar for
     * the object. A side effect of this is that there may be different Avatars returned for different identifiers for
     * the same domain object. One example of this is support for multiple email addresses. If the AvatarProvider usees
     * Gravatar, there could be different avatar images registered for each of the different email addresses for the
     * user. The relationship of identifiers to users is surjective.
     *
     * @return the identifier!
     */
    String getIdentifier();

    /**
     * Whether or not to use the default/unknown avatar. Note that an AvatarProvider may not know if the "mystery man"
     * or default avatar is being used. Gravatar provides a default image which will be returned if there is no
     * Gravatar registered but unless you fetch the image yourself, you can't know if that's actually what happened.
     * Other implementations of AvatarProvider handle this logic.
     *
     * @return true only if the unknown avatar must be used for this owner.
     */
    boolean useUnknownAvatar();

    /**
     * Get the instance of the domain object that owns the avatar.
     *
     * @return the owner domain object.
     */
    T get();

}
