package com.atlassian.plugins.whitelist.ui;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import com.atlassian.plugins.whitelist.ImmutableWhitelistRule;
import com.atlassian.plugins.whitelist.WhitelistRule;
import com.atlassian.plugins.whitelist.WhitelistType;

import com.google.common.base.Objects;

import org.codehaus.jackson.annotate.JsonAutoDetect;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonIgnoreProperties;
import org.codehaus.jackson.annotate.JsonProperty;

import static com.google.common.base.Preconditions.checkNotNull;

@Immutable
@JsonAutoDetect
@JsonIgnoreProperties(ignoreUnknown=true)
public final class WhitelistBean
{
    private final Integer id;
    private final String expression;
    private final WhitelistType type;
    private final Boolean allowInbound;
    private final String iconUrl;

    @JsonCreator
    @SuppressWarnings("UnusedDeclaration")
    public WhitelistBean(@JsonProperty("expression") final String expression, @JsonProperty("type") final String type, @JsonProperty("allowInbound") final Boolean allowInbound)
    {
        this.id = null;
        this.expression = expression;
        this.type = type != null ? WhitelistTypeMapper.asType(type) : null;
        this.iconUrl = null;
        this.allowInbound = allowInbound;
    }

    public WhitelistBean(final WhitelistBeanBuilder builder)
    {
        checkNotNull(builder, "builder");
        this.id = builder.id;
        this.expression = builder.expression;
        this.type = builder.type;
        this.iconUrl = builder.iconUrl;
        this.allowInbound = builder.allowInbound;
    }

    public static WhitelistBeanBuilder builder()
    {
        return new WhitelistBeanBuilder();
    }

    public WhitelistRule asRule()
    {
        return ImmutableWhitelistRule.builder()
                .id(id)
                .expression(expression)
                .type(type)
                .allowInbound(allowInbound)
                .build();
    }

    /**
     * @param whitelistRule fill up any null values with the information contained in the given whitelist rule
     * @return a new whitelist bean where all null values are updated with the ones from the whitelist rule
     */
    public WhitelistRule populateWith(final WhitelistRule whitelistRule)
    {
        checkNotNull(whitelistRule, "whitelistRule");
        final String newExpression = Objects.firstNonNull(expression, whitelistRule.getExpression());
        final WhitelistType newType = Objects.firstNonNull(type, whitelistRule.getType());
        final boolean newAllowInbound = Objects.firstNonNull(allowInbound, whitelistRule.isAllowInbound());
        return ImmutableWhitelistRule.builder()
                                     .id(whitelistRule.getId())
                                     .expression(newExpression)
                                     .type(newType)
                                     .allowInbound(newAllowInbound)
                                     .build();
    }

    @Nullable
    public Integer getId()
    {
        return id;
    }

    /**
     * @return the actual expression to be shown on the ui; in terms of applinks, this is <em>not</em> the expression
     * stored in the database.
     */
    public String getExpression()
    {
        return expression;
    }

    @JsonIgnore
    public WhitelistType getType()
    {
        return type;
    }

    @SuppressWarnings("UnusedDeclaration")
    @JsonProperty("type")
    public String getTypeAsString()
    {
        return WhitelistTypeMapper.asString(type);
    }

    public String getIconUrl()
    {
        return iconUrl;
    }

    public Boolean isAllowInbound()
    {
        return allowInbound;
    }

    @SuppressWarnings("UnusedDeclaration")
    public boolean isAllowEdit()
    {
        return type != WhitelistType.APPLICATION_LINK;
    }

    @SuppressWarnings("UnusedDeclaration")
    public boolean isAllowDelete()
    {
        return type != WhitelistType.APPLICATION_LINK;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o)
        {
            return true;
        }
        if (o == null || getClass() != o.getClass())
        {
            return false;
        }

        final WhitelistBean that = (WhitelistBean) o;
        return Objects.equal(id, that.id) &&
                Objects.equal(expression, that.expression) &&
                Objects.equal(type, that.type);
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(id, expression, type);
    }

    @Override
    public String toString()
    {
        return "WhitelistBean{id=" + id + ", expression='" + expression + '\'' + ", type=" + type + '}';
    }
}
