package com.atlassian.plugins.whitelist.ui;

import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import com.atlassian.plugins.whitelist.InboundWhitelist;
import com.atlassian.plugins.whitelist.OutboundWhitelist;
import com.atlassian.plugins.whitelist.WhitelistService;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.websudo.WebSudoRequired;
import com.google.common.collect.ImmutableMap;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.regex.PatternSyntaxException;

@Path("/")
@Consumes({ MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_JSON })
public class WhitelistResource
{
    private final WhitelistService whitelistService;
    private final I18nResolver i18nResolver;
    private final OutboundWhitelist outboundWhitelist;
    private final InboundWhitelist inboundWhitelist;
    private final WhitelistBeanService whitelistBeanService;

    public WhitelistResource(final WhitelistService whitelistService, final I18nResolver i18nResolver, final OutboundWhitelist outboundWhitelist, final InboundWhitelist inboundWhitelist, final WhitelistBeanService whitelistBeanService)
    {
        this.whitelistService = whitelistService;
        this.i18nResolver = i18nResolver;
        this.outboundWhitelist = outboundWhitelist;
        this.inboundWhitelist = inboundWhitelist;
        this.whitelistBeanService = whitelistBeanService;
    }

    @Path("enable")
    @POST
    @WebSudoRequired
    public Response enable()
    {
        whitelistService.enableWhitelist();
        return Response.ok().build();
    }

    @Path("disable")
    @POST
    @WebSudoRequired
    public Response disable()
    {
        whitelistService.disableWhitelist();
        return Response.ok().build();
    }

    @Path("check")
    @GET
    @AnonymousAllowed
    public Response isAllowed(@QueryParam("url") final String uriString)
    {
        try
        {
            URI uri = new URI(uriString);
            return Response.ok(ImmutableMap.of(
                    "outbound", outboundWhitelist.isAllowed(uri),
                    "inbound", inboundWhitelist.isAllowed(uri)
            )).build();
        }
        catch (URISyntaxException e)
        {
            return error(e.getMessage());
        }
    }

    @POST
    @WebSudoRequired
    public Response create(final WhitelistBean whitelistBean)
    {
        if (whitelistBean.getExpression() == null)
        {
            return fieldError("expression", i18nResolver.getText("whitelist.ui.error.noexpression"));
        }
        try
        {
            final WhitelistBean result = whitelistBeanService.add(whitelistBean);
            return Response.ok(result).build();
        }
        catch (PatternSyntaxException e)
        {
            return fieldError("expression", e.getMessage());
        }
        catch (IllegalArgumentException e)
        {
            return error(e.getMessage());
        }
    }

    @PUT
    @Path("{id}")
    @WebSudoRequired
    public Response update(@PathParam("id") final int id, final WhitelistBean whitelistBean)
    {
        try
        {
            final WhitelistBean result = whitelistBeanService.update(id, whitelistBean);
            return Response.ok(result).build();
        }
        catch (NullPointerException e)
        {
            return error(i18nResolver.getText("whitelist.ui.error.deleted"));
        }
        catch (IllegalArgumentException e)
        {
            return error(e.getMessage());
        }
    }

    @DELETE
    @Path("{id}")
    @WebSudoRequired
    public Response delete(@PathParam("id") final int id)
    {
        try
        {
            whitelistService.remove(id);
        }
        catch (NullPointerException e)
        {
        }
        catch (IllegalArgumentException e)
        {
            return error(e.getMessage());
        }
        return Response.ok().build();
    }

    private static Response fieldError(final String field, final String message)
    {
        return Response.status(Response.Status.BAD_REQUEST).entity(ImmutableMap.of("errors", ImmutableMap.of(field, message))).build();
    }

    private static Response error(final String message)
    {
        return Response.status(Response.Status.BAD_REQUEST).entity(ImmutableMap.of("message", message)).build();
    }
}
