package com.atlassian.plugins.whitelist.migration.confluence.macros;

import com.atlassian.bandana.BandanaManager;
import com.atlassian.plugins.whitelist.WhitelistManager;
import com.atlassian.plugins.whitelist.WhitelistOnOffSwitch;
import com.atlassian.plugins.whitelist.WhitelistRule;
import com.atlassian.plugins.whitelist.migration.AbstractWhitelistPluginUpgradeTask;
import com.atlassian.plugins.whitelist.migration.confluence.CustomConfluenceBandanaContext;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.atlassian.sal.api.upgrade.PluginUpgradeTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Collections;

/**
 * Migrates existing whitelist entries used by the RSS and the HTML include macro. Doesn't migrate the whitelist used
 * by Confluence Gadgets.
 * @since 1.0
 */
public class ConfluenceMacroWhitelistMigration extends AbstractWhitelistPluginUpgradeTask implements PluginUpgradeTask
{
    private static final String MACRO_WHITELIST_BANDANA_KEY = "com.atlassian.allowed.domains";
    private static final Logger logger = LoggerFactory.getLogger(ConfluenceMacroWhitelistMigration.class);

    private final BandanaManager bandanaManager;
    private final WhitelistOnOffSwitch whitelistOnOffSwitch;
    private final WhitelistManager whitelistManager;
    private final TransactionTemplate transactionTemplate;
    private final BandanaMacroWhitelistXmlParser parser;

    public ConfluenceMacroWhitelistMigration(final BandanaManager bandanaManager, final WhitelistOnOffSwitch whitelistOnOffSwitch, final WhitelistManager whitelistManager, final TransactionTemplate transactionTemplate)
    {
        this.bandanaManager = bandanaManager;
        this.whitelistOnOffSwitch = whitelistOnOffSwitch;
        this.whitelistManager = whitelistManager;
        this.transactionTemplate = transactionTemplate;
        this.parser = new BandanaMacroWhitelistXmlParser();
    }

    @Override
    public int getBuildNumber()
    {
        return 2;
    }

    @Override
    public String getShortDescription()
    {
        return "Migrate existing whitelist entries used by the RSS and HTML include macro.";
    }

    @Override
    public Collection<Message> doUpgrade() throws Exception
    {
        return transactionTemplate.execute(new TransactionCallback<Collection<Message>>()
        {
            @Override
            public Collection<Message> doInTransaction()
            {
                final Object value = bandanaManager.getValue(new CustomConfluenceBandanaContext(), MACRO_WHITELIST_BANDANA_KEY);
                if (value instanceof String)
                {
                    migrateExistingData((String) value);
                }
                return Collections.emptyList();
            }
        });
    }

    private void migrateExistingData(final String bandanaContent)
    {
        final BandanaMacroWhitelistXmlData data = parser.parseData(bandanaContent);
        migrateWhitelistRules(data);
        migrateOnOffState(data);
    }

    private void migrateWhitelistRules(final BandanaMacroWhitelistXmlData data)
    {
        final Collection<WhitelistRule> whitelistRules = data.getRules();
        logger.debug("Migrating {} whitelist rules ...", whitelistRules.size());
        whitelistManager.addAll(whitelistRules);
    }

    private void migrateOnOffState(final BandanaMacroWhitelistXmlData data)
    {
        if (data.isAllAllowed()) {
            logger.debug("Confluence macro whitelist was disabled");
            whitelistOnOffSwitch.disable();
        }
        else {
            logger.debug("Confluence macro whitelist was enabled");
            whitelistOnOffSwitch.enable();
        }
    }

}
