package com.atlassian.plugins.whitelist.matcher;

import com.google.common.base.Predicate;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.regex.Pattern;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @since 1.0
 */
public class WildcardExpressionMatcher implements Predicate<URI>
{
    private static final String[] RULE_ESCAPE_CHARACTERS = { ".", "?", "+", "|" };
    private static final String WILDCARD_CHARACTER_PATTERN = ".*";

    private Pattern pattern;

    public WildcardExpressionMatcher(final String expression)
    {
        pattern = Pattern.compile(createRegex(checkNotNull(expression, "expression")), Pattern.CASE_INSENSITIVE);
    }

    @Override
    public boolean apply(final URI uri)
    {
        try
        {
            final URI normalizedUri = MatcherUtils.normalizeUri(uri);
            return pattern.matcher(normalizedUri.toString()).matches();
        }
        catch (URISyntaxException e)
        {
            return false;
        }
    }

    private static String createRegex(final String rule)
    {
        String regex = rule;
        for (String escapeChar : RULE_ESCAPE_CHARACTERS)
        {
            regex = regex.replaceAll("\\" + escapeChar, "\\\\" + escapeChar);
        }
        return regex.replaceAll("\\*", WILDCARD_CHARACTER_PATTERN);
    }
}
