package com.atlassian.plugins.whitelist;

import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugins.whitelist.events.ClearWhitelistCacheEvent;
import com.atlassian.plugins.whitelist.events.WhitelistRuleAddedEvent;
import com.atlassian.plugins.whitelist.events.WhitelistRuleChangedEvent;
import com.atlassian.plugins.whitelist.events.WhitelistRuleRemovedEvent;
import com.atlassian.plugins.whitelist.permission.PermissionChecker;
import com.google.common.base.Supplier;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Nullable;
import java.util.Collection;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Does permission checkking and keeps a cache of known whitelist rules.
 * @since 1.1
 */
public class WhitelistServiceImpl implements WhitelistService, InitializingBean, DisposableBean
{
    private final WhitelistOnOffSwitch whitelistOnOffSwitch;
    private final WhitelistManager whitelistManager;
    private final PermissionChecker permissionChecker;
    private final EventPublisher eventPublisher;
    // TODO: update the one-entry cache solution as soon as Guava has been upgraded to 11.x
    private final Cache<Object, Collection<WhitelistRule>> cache = CacheBuilder.newBuilder().build(CacheLoader.from(new Supplier<Collection<WhitelistRule>>()
    {
        @Override
        public Collection<WhitelistRule> get()
        {
            return ImmutableList.copyOf(Iterables.transform(whitelistManager.getAll(), ImmutableWhitelistRuleBuilder.COPY));
        }
    }));

    public WhitelistServiceImpl(final WhitelistOnOffSwitch whitelistOnOffSwitch, final WhitelistManager whitelistManager, final PermissionChecker permissionChecker, final EventPublisher eventPublisher)
    {
        this.whitelistOnOffSwitch = whitelistOnOffSwitch;
        this.whitelistManager = whitelistManager;
        this.permissionChecker = permissionChecker;
        this.eventPublisher = eventPublisher;
    }

    @Override
    public void afterPropertiesSet() throws Exception
    {
        eventPublisher.register(this);
    }

    @Override
    public void destroy() throws Exception
    {
        eventPublisher.unregister(this);
    }

    @Override
    public boolean isWhitelistEnabled()
    {
        return whitelistOnOffSwitch.isEnabled();
    }

    @Override
    public void enableWhitelist()
    {
        permissionChecker.checkCurrentUserCanManageWhitelist();
        whitelistOnOffSwitch.enable();
    }

    @Override
    public void disableWhitelist()
    {
        permissionChecker.checkCurrentUserCanManageWhitelist();
        whitelistOnOffSwitch.disable();
    }

    @Override
    public WhitelistRule add(final WhitelistRule whitelistRule)
    {
        checkNotNull(whitelistRule, "whitelistRule");
        permissionChecker.checkCurrentUserCanManageWhitelist();
        checkTypeNotApplicationLink(whitelistRule);
        return whitelistManager.add(whitelistRule);
    }

    @Override
    public WhitelistRule update(final WhitelistRule whitelistRule)
    {
        checkNotNull(whitelistRule, "whitelistRule");
        final Integer whitelistRuleId = whitelistRule.getId();
        checkArgument(whitelistRuleId != null, "Cannot update whitelist rule, the given instance has no database id: " + whitelistRule);
        permissionChecker.checkCurrentUserCanManageWhitelist();
        enforceApplicationLinkUpdateRules(whitelistRule);
        return whitelistManager.update(whitelistRule);
    }

    @Override
    public void remove(final int id)
    {
        permissionChecker.checkCurrentUserCanManageWhitelist();
        final WhitelistRule whitelistRule = get(id);
        if (whitelistRule != null)
        {
            checkTypeNotApplicationLink(whitelistRule);
            whitelistManager.remove(whitelistRule);
        }
    }

    @Override
    public Collection<WhitelistRule> getAll()
    {
        return cache.getUnchecked(this);
    }

    @Nullable
    @Override
    public WhitelistRule get(final int id)
    {
        return Iterables.find(cache.getUnchecked(this), WhitelistRulePredicates.withId(id), null);
    }

    @EventListener
    @SuppressWarnings("UnusedParameters")
    public void onWhitelistRuleAddedEvent(final WhitelistRuleAddedEvent event)
    {
        cache.invalidateAll();
    }

    @EventListener
    @SuppressWarnings("UnusedParameters")
    public void onWhitelistRuleChanged(final WhitelistRuleChangedEvent event)
    {
        cache.invalidateAll();
    }

    @EventListener
    @SuppressWarnings("UnusedParameters")
    public void onWhitelistRuleRemoved(final WhitelistRuleRemovedEvent event)
    {
        cache.invalidateAll();
    }

    @EventListener
    @SuppressWarnings("UnusedParameters")
    public void onClearWhitelistCacheEvent(final ClearWhitelistCacheEvent event)
    {
        cache.invalidateAll();
    }

    private static void checkTypeNotApplicationLink(final WhitelistRule whitelistRule)
    {
        if (whitelistRule.getType() == WhitelistType.APPLICATION_LINK)
        {
            throw new IllegalArgumentException("Adding new application link whitelist rules is not supported, they are managed internally.");
        }
    }

    @SuppressWarnings("ConstantConditions")
    private void enforceApplicationLinkUpdateRules(final WhitelistRule whitelistRule)
    {
        final Integer whitelistRuleId = whitelistRule.getId();
        checkNotNull(whitelistRuleId, "whitelistRuleId");
        final WhitelistRule existingWhitelistRule = get(whitelistRuleId);
        if (existingWhitelistRule == null)
        {
            throw new WhitelistRuleNotFoundException("Whitelist rule with id '" + whitelistRuleId + "' not found.");
        }
        if (existingWhitelistRule.getType() == WhitelistType.APPLICATION_LINK)
        {
            checkArgument(whitelistRule.getType() == WhitelistType.APPLICATION_LINK && whitelistRule.getExpression().equals(existingWhitelistRule.getExpression()), "Cannot change the type or expression of this application link whitelist rule.");
        }
        else
        {
            checkArgument(whitelistRule.getType() != WhitelistType.APPLICATION_LINK, "Cannot change the type to application link.");
        }
    }

}
