package com.atlassian.plugins.whitelist;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nullable;
import java.util.Collection;

/**
 * Handles all whitelist rules related data queries and modification.
 * @since 1.0
 */
@PublicApi
public interface WhitelistManager
{
    /**
     * Add a new whitelist rule. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistRuleAddedEvent} after
     * the modification has been finished.
     * @param whitelistRule the whitelist rule to be added
     * @return the created whitelist rule
     */
    WhitelistRule add(WhitelistRule whitelistRule);

    /**
     * Add the given collection of whitelist rules. Publishes a {@link com.atlassian.plugins.whitelist.events.WhitelistRuleAddedEvent}
     * for each rule which was added after all rules have been processed.
     * @param whitelistRules the collection of whitelist rules to be added
     * @return the newly created whitelist rules
     */
    Collection<WhitelistRule> addAll(Iterable<WhitelistRule> whitelistRules);

    /**
     * Update a given whitelist rule. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistRuleChangedEvent}
     * after the modification has been finished.
     * @param whitelistRule the updated information to be used with the whitelist rule
     * @return the updated whitelist rule
     */
    WhitelistRule update(WhitelistRule whitelistRule);

    /**
     * Remove the given whitelist rule. Publishes {@link com.atlassian.plugins.whitelist.events.WhitelistRuleRemovedEvent}
     * after the modification has been finished.
     * @param whitelistRule the whitelistRule to be removed
     */
    void remove(WhitelistRule whitelistRule);

    /**
     * Remove the given collection of whitelist rules. Published a {@link com.atlassian.plugins.whitelist.events.WhitelistRuleRemovedEvent}
     * for each rule that was removed after all rules have been processed.
     * @param whitelistRules the collection of whitelist rules to be removed
     */
    void removeAll(Iterable<WhitelistRule> whitelistRules);

    /**
     * @return all known whitelist rules
     */
    Collection<WhitelistRule> getAll();

    /**
     * @param id the id of the whitelist rule
     * @return the whitelist rule with the given id or <code>null</code> if not found
     */
    @Nullable
    WhitelistRule get(int id);
}
