package com.atlassian.plugins.whitelist;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.plugins.whitelist.applinks.ApplicationLinkMatcher;
import com.atlassian.plugins.whitelist.applinks.CachingApplicationLinkService;
import com.atlassian.plugins.whitelist.matcher.DomainNameMatcher;
import com.atlassian.plugins.whitelist.matcher.ExactUrlMatcher;
import com.atlassian.plugins.whitelist.matcher.RegularExpressionMatcher;
import com.atlassian.plugins.whitelist.matcher.WildcardExpressionMatcher;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @since 1.0
 */
class DefaultWhitelistRuleMatcher implements Predicate<URI>
{
    private static final Logger logger = LoggerFactory.getLogger(DefaultWhitelistRuleMatcher.class);

    private final CachingApplicationLinkService applicationLinkService;
    private final WhitelistRule whitelistRule;
    private final ImmutableMap<WhitelistType, Function<WhitelistRule, Predicate<URI>>> mappings = ImmutableMap.of(
            WhitelistType.APPLICATION_LINK, createApplicationLinkRule(),
            WhitelistType.EXACT_URL, createExactUrlRule(),
            WhitelistType.WILDCARD_EXPRESSION, createWildcardExpressionRule(),
            WhitelistType.REGULAR_EXPRESSION, createRegularExpressionRule(),
            WhitelistType.DOMAIN_NAME, createDomainNameRule()
    );

    public DefaultWhitelistRuleMatcher(final CachingApplicationLinkService applicationLinkService, final WhitelistRule whitelistRule)
    {
        this.applicationLinkService = checkNotNull(applicationLinkService, "applicationLinkService");
        this.whitelistRule = whitelistRule;
    }

    /**
     * @param uri the uri to be matched
     * @return <code>true</code> if the given uri matches the given rule; <code>false</code> otherwise
     */
    public boolean apply(final URI uri)
    {
        checkNotNull(whitelistRule, "whitelistRule");
        final WhitelistType type = whitelistRule.getType();
        final Function<WhitelistRule, Predicate<URI>> mapping = mappings.get(type);
        if (mapping == null)
        {
            logger.debug("No mapping found for whitelist type '{}', ignoring data '{}'.", type, whitelistRule);
            return false;
        }
        else
        {
            try
            {
                return mapping.apply(whitelistRule).apply(uri);
            }
            catch (RuntimeException e)
            {
                logger.debug("Failed to match '{}' with whitelist rule '{}'", new Object[] {uri, whitelistRule, e});
                return false;
            }
        }
    }

    private Function<WhitelistRule, Predicate<URI>> createApplicationLinkRule()
    {
        return new Function<WhitelistRule, Predicate<URI>>() {

            @Override
            public Predicate<URI> apply(final WhitelistRule input)
            {
                final String applicationId = input.getExpression();
                final ApplicationLink applicationLink = applicationLinkService.getApplicationLink(new ApplicationId(applicationId));
                if (applicationLink == null)
                {
                    throw new IllegalArgumentException("Failed to resolved application link with application id '" + applicationId + "'; may be it has been removed and the whitelist was not updated?");
                }
                else
                {
                    return createWhitelistRule(applicationLink);
                }
            }

            private Predicate<URI> createWhitelistRule(final ApplicationLink applicationLink)
            {
                return new ApplicationLinkMatcher(applicationLink);
            }
        };
    }

    private Function<WhitelistRule, Predicate<URI>> createExactUrlRule()
    {
        return new Function<WhitelistRule, Predicate<URI>>()
        {
            @Override
            public Predicate<URI> apply(final WhitelistRule input)
            {
                return new ExactUrlMatcher(input.getExpression());
            }
        };
    }

    private Function<WhitelistRule, Predicate<URI>> createWildcardExpressionRule()
    {
        return new Function<WhitelistRule, Predicate<URI>>()
        {
            @Override
            public Predicate<URI> apply(final WhitelistRule input)
            {
                return new WildcardExpressionMatcher(input.getExpression());
            }
        };
    }

    private Function<WhitelistRule, Predicate<URI>> createRegularExpressionRule()
    {
        return new Function<WhitelistRule, Predicate<URI>>()
        {
            @Override
            public Predicate<URI> apply(final WhitelistRule input)
            {
                return new RegularExpressionMatcher(input.getExpression());
            }
        };
    }

    private Function<WhitelistRule, Predicate<URI>> createDomainNameRule()
    {
        return new Function<WhitelistRule, Predicate<URI>>()
        {
            @Override
            public Predicate<URI> apply(final WhitelistRule input)
            {
                return new DomainNameMatcher(input.getExpression());
            }
        };
    }
}
