package com.atlassian.plugins.whitelist;

import com.atlassian.plugins.whitelist.applinks.CachingApplicationLinkService;
import com.atlassian.plugins.whitelist.matcher.SelfUrlMatcher;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import javax.annotation.Nullable;
import java.net.URI;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @since 1.1
 */
abstract class AbstractWhitelist implements Whitelist
{
    private final WhitelistOnOffSwitch whitelistOnOffSwitch;
    private final WhitelistService whitelistService;
    private final SelfUrlMatcher selfUrlMatcher;
    private final CachingApplicationLinkService applicationLinkService;

    public AbstractWhitelist(final WhitelistOnOffSwitch whitelistOnOffSwitch, final WhitelistService whitelistService, final SelfUrlMatcher selfUrlMatcher, final CachingApplicationLinkService applicationLinkService)
    {
        this.whitelistOnOffSwitch = whitelistOnOffSwitch;
        this.whitelistService = whitelistService;
        this.selfUrlMatcher = selfUrlMatcher;
        this.applicationLinkService = applicationLinkService;
    }

    @Override
    public boolean isAllowed(final URI uri)
    {
        checkNotNull(uri, "uri");
        return isWhitelistDisabled() || selfUrlMatcher.apply(uri) || ruleMatchingUrlExists(uri);
    }

    private boolean isWhitelistDisabled()
    {
        return !whitelistOnOffSwitch.isEnabled();
    }

    private boolean ruleMatchingUrlExists(final URI uri)
    {
        return Iterables.find(whitelistService.getAll(), matchesUrl(uri), null) != null;
    }

    private Predicate<WhitelistRule> matchesUrl(final URI uri)
    {
        return new Predicate<WhitelistRule>()
        {
            @Override
            public boolean apply(@Nullable final WhitelistRule input)
            {
                return input != null && matches(input, uri);
            }
        };
    }

    protected boolean matches(final WhitelistRule whitelistRule, final URI uri)
    {
        return new DefaultWhitelistRuleMatcher(applicationLinkService, whitelistRule).apply(uri);
    }
}
