package com.atlassian.plugin.webresource.impl.support;

import com.atlassian.plugin.webresource.impl.snapshot.Bundle;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import org.apache.commons.io.IOUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.SocketException;
import java.util.List;

/**
 * Basic utility functions.
 *
 * @since v6.3
 */
public class Support
{
    public static final Logger LOGGER = LoggerFactory.getLogger("webresource");

    /**
     * Join list of predicates with AND operand.
     */
    public static Predicate<Bundle> efficientAndPredicate(List<Predicate<Bundle>> predicates)
    {
        if (predicates.size() == 0)
        {
            return Predicates.<Bundle>alwaysTrue();
        }
        else if (predicates.size() == 1)
        {
            return predicates.get(0);
        }
        else
        {
            return Predicates.and(predicates);
        }
    }

    /**
     * Indent string, multiline also supported.
     */
    public static String indent(String string, String indent)
    {
        return indent + string.replaceAll("\\n", "\n" + indent);
    }

    /**
     * Helper to check for equality without bothering to handle nulls.
     */
    public static boolean equals(Object a, Object b)
    {
        return a == null ? b == null : a.equals(b);
    }

    /**
     * Copy in into out and close streams safely.
     */
    public static void copy(final InputStream in, final OutputStream out)
    {
        try
        {
            IOUtils.copy(in, out);
        }
        catch (final IOException e)
        {
            logIOException(e);
        }
        finally
        {
            IOUtils.closeQuietly(in);
            try
            {
                out.flush();
            }
            catch (final IOException e)
            {
                Support.LOGGER.debug("Error flushing output stream", e);
            }
        }
    }

    public static void logIOException(IOException e){
        if (e instanceof SocketException && "Broken pipe".equals(e.getMessage())) {
            Support.LOGGER.trace("Client Abort error", e);
        } else {
            Support.LOGGER.debug("IO Exception", e);
        }
    }

    /**
     * Parses XML.
     */
    public static Element parseXml(String xml)
    {
        Document document;
        try
        {
            document = DocumentHelper.parseText(xml);
        }
        catch (DocumentException e)
        {
            throw new RuntimeException(e);
        }
        return document.getRootElement();
    }
}
