package com.atlassian.plugin.webresource.assembler;

import com.google.common.base.Objects;
import com.google.common.collect.Sets;

import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Mutable representation of the current state of resources included.
 * @since 3.2
 */
class InclusionState
{
    /** Webresources that have been included in previous calls to includeResources, and all the individual resources
     * in included contexts */
    private Set<String> webresources;
    /** Webresource contexts that have been included in previous calls to includeResources */
    private Set<String> contexts;
    /** Data keys that have been included in previous calls to includeData */
    private Set<String> dataKeys;

    public InclusionState(Set<String> webresources, Set<String> contexts, Set<String> dataKeys)
    {
        checkNotNull(webresources);
        checkNotNull(contexts);
        checkNotNull(dataKeys);

        this.webresources = webresources;
        this.contexts = contexts;
        this.dataKeys = dataKeys;
    }

    public InclusionState copy()
    {
        return new InclusionState(Sets.newHashSet(webresources), Sets.newHashSet(contexts),
                Sets.newHashSet(dataKeys));
    }

    public void reset(InclusionState that)
    {
        this.webresources = Sets.newHashSet(that.webresources);
        this.contexts = Sets.newHashSet(that.contexts);
        this.dataKeys = Sets.newHashSet(that.dataKeys);
    }

    /**
     * Webresources that have been included in previous calls to includeResources, and all the individual resources
     * in included contexts. Mutable!
     */
    public Set<String> getWebresources()
    {
        return webresources;
    }

    /**
     * Webresource contexts that have been included in previous calls to includeResources. Mutable!
     */
    public Set<String> getContexts()
    {
        return contexts;
    }

    /**
     * Data keys that have been included in previous calls to includeData. Mutable!
     */
    public Set<String> getDataKeys()
    {
        return dataKeys;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        InclusionState that = (InclusionState) o;

        if (!contexts.equals(that.contexts)) return false;
        if (!dataKeys.equals(that.dataKeys)) return false;
        if (!webresources.equals(that.webresources)) return false;

        return true;
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(webresources, contexts, dataKeys);
    }
}
