package com.atlassian.plugin.webresource.assembler;

import com.atlassian.plugin.event.PluginEventListener;
import com.atlassian.plugin.event.events.PluginDisabledEvent;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.plugin.event.events.PluginModuleDisabledEvent;
import com.atlassian.plugin.event.events.PluginModuleEnabledEvent;
import com.atlassian.plugin.webresource.Globals;
import com.atlassian.plugin.webresource.PluginResourceLocator;
import com.atlassian.plugin.webresource.ResourceBatchingConfiguration;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.WebResourceModuleDescriptor;
import com.atlassian.plugin.webresource.WebResourceUrlProvider;
import com.atlassian.plugin.webresource.transformer.StaticTransformers;
import com.atlassian.webresource.api.assembler.WebResourceAssemblerBuilder;
import com.atlassian.webresource.api.assembler.WebResourceAssemblerFactory;
import com.google.common.annotations.VisibleForTesting;

/**
 * Default implementation of WebResourceAssemblerFactory
 * @since v3.0
 */
public class DefaultWebResourceAssemblerFactory implements WebResourceAssemblerFactory
{
    private final ResourceBatchingConfiguration batchingConfiguration;
    private final WebResourceIntegration webResourceIntegration;
    private final PluginResourceLocator pluginResourceLocator;
    private final StaticTransformers staticTransformers;
    private final WebResourceSetCache webResourceSetCache;

    public DefaultWebResourceAssemblerFactory(
            ResourceBatchingConfiguration batchingConfiguration,
            WebResourceIntegration webResourceIntegration, WebResourceUrlProvider webResourceUrlProvider,
            PluginResourceLocator pluginResourceLocator, StaticTransformers staticTransformers)
    {
        this.batchingConfiguration = batchingConfiguration;
        this.webResourceIntegration = webResourceIntegration;
        this.pluginResourceLocator = pluginResourceLocator;
        this.staticTransformers = staticTransformers;
        webResourceSetCache = new WebResourceSetCache(pluginResourceLocator.temporaryWayToGetGlobalsDoNotUseIt(), webResourceIntegration.getCacheFactory(),
            staticTransformers);

        // TODO find better way to register, move it outside of the constructor.
        webResourceIntegration.getPluginEventManager().register(this);
    }

    @Override
    public WebResourceAssemblerBuilder create()
    {
        return new DefaultWebResourceAssemblerBuilder(batchingConfiguration, webResourceIntegration, pluginResourceLocator, staticTransformers, webResourceSetCache);
    }

    @VisibleForTesting
    WebResourceSetCache getWebResourceSetCache()
    {
        return webResourceSetCache;
    }

    @Override
    public void clearCache()
    {
        webResourceSetCache.clear();
    }

    @PluginEventListener
    public void onPluginDisabled(final PluginDisabledEvent event)
    {
        webResourceSetCache.clear();
    }

    @PluginEventListener
    public void onPluginEnabled(final PluginEnabledEvent event)
    {
        webResourceSetCache.clear();
    }

    @PluginEventListener
    public void onPluginModuleEnabled(final PluginModuleEnabledEvent event)
    {
        if (event.getModule() instanceof WebResourceModuleDescriptor)
            webResourceSetCache.clear();
    }

    @PluginEventListener
    public void onPluginModuleDisabled(final PluginModuleDisabledEvent event)
    {
        if (event.getModule() instanceof WebResourceModuleDescriptor)
            webResourceSetCache.clear();
    }
}
