package com.atlassian.plugin.webresource.url;

/**
 * Describes whether a resource can be safely delivered via a stateless route like a CDN.
 * Such routes will drop cookies (CDNs do this, as to cookie-less domains), etc. This means
 * that any state the resource is relying on (e.g. current user) will not be preserved
 * between page-generation-time and request-delivery-time.
 *
 * Typically, a resource is safe for CDN delivery if non of the transforms are legacy transforms,
 * and none of the conditions are legacy conditions. This means using UrlReadingWebResourceTransformer and
 * UrlReadingCondition, which capture their state in the url and are hence otherwise stateless.
 *
 * @since 3.0.5
 */
public enum CDNSupport {
    /** something in this resource is not stateless, and the resource cannot be delivered via CDN. */
    Tainted("-T"),
    /** this resource is stateless */
    OK("-CDN");

    private final String suffix;

    private CDNSupport(String suffix)
    {
        this.suffix = suffix;
    }

    /**
     * @return Tainted if either this or the other is tainted, OK otherwise
     */
    public CDNSupport combine(CDNSupport other) {
        return this == Tainted || other == Tainted ? Tainted : OK;
    }

    /**
     * @return the suffix to add to the hash, to make it visible to devs if a resource
     * is CDN-safe or not.
     */
    public String hashSuffix() {
        // make CDN tainted-ness visible in urls by devs as a migration aid
        return suffix;
    }
}