package com.atlassian.plugin.webresource.cache;

import com.atlassian.plugin.cache.filecache.FileCache;
import com.atlassian.plugin.cache.filecache.FileCacheStreamProvider;
import com.atlassian.plugin.servlet.DownloadException;
import com.atlassian.plugin.webresource.ResourceUtils;

import java.io.OutputStream;
import java.util.Map;
import java.util.TreeMap;

/**
 * A small facade around a FileCacheStreamProvider, that captures both the cache key
 * and FileCache, saving having to pass them around separately
 *
 * @since v2.13
 */
public interface CacheHandle {
    void stream(OutputStream out, FileCacheStreamProvider streamProvider) throws DownloadException;

    public static class Builder {
        /** does not cache, calls the stream provider directly */
        public static CacheHandle passthrough() {
            return new CacheHandle() {
                @Override
                public void stream(OutputStream out, FileCacheStreamProvider streamProvider) throws DownloadException {
                    streamProvider.writeStream(out);
                }
            };
        }

        /**
         * inspects the request, and if appropriate, returns a CacheHandle that will
         * cache resources to the cache.
         * If not approriate, a passthrough() handle is returned.
         */
        public static CacheHandle forRequest(final FileCache<FileCacheKey> cache, String downloadtype, String path, final Map<String, String> params)
        {
            if (!ResourceUtils.canRequestedResourcesContentBeAssumedConstant(params))
            {
                return passthrough();
            }

            final FileCacheKey cacheKey = new FileCacheKey(new TreeMap<String, String>(params), downloadtype, path);
            return new CacheHandle() {
                @Override
                public void stream(OutputStream out, FileCacheStreamProvider streamProvider) throws DownloadException {
                    cache.stream(cacheKey, out, streamProvider);
                }
            };
        }

    }

}
