package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.url.UrlParameters;

import java.util.HashSet;
import java.util.Map;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.servlet.AbstractFileServerServlet.SERVLET_PATH;

/**
 * Creates a batch of all like-typed resources that are declared as "super-batch="true"" in their plugin
 * definitions.
 * <p/>
 * The URL for batch resources is /download/superbatch/&lt;type>/batch.&lt;type. The additional type part in the path
 * is simply there to make the number of path-parts identical with other resources, so relative URLs will still work
 * in CSS files.
 */
public class SuperBatchPluginResource extends AbstractPluginResource
{
    static final String URL_PREFIX = PATH_SEPARATOR + SERVLET_PATH + PATH_SEPARATOR + "superbatch" + PATH_SEPARATOR;
    static final String DEFAULT_RESOURCE_NAME_PREFIX = "batch";

    private final ResourceKey resource;

    public static SuperBatchPluginResource createBatchFor(final PluginResource pluginResource)
    {
        return new SuperBatchPluginResource(pluginResource.getType(), pluginResource.getUrlParameters(), pluginResource.getParams());
    }

    public SuperBatchPluginResource(final String type, final UrlParameters urlParams, final Map<String, String> params)
    {
        this(ResourceKey.Builder.batch(type), urlParams, params);
    }

    protected SuperBatchPluginResource(final ResourceKey resource, final UrlParameters urlParams, final Map<String, String> params)
    {
        super(urlParams, params, new HashSet<BatchedWebResourceDescriptor>());
        this.resource = resource;
    }

    @Override
    public String getUrl()
    {
        final StringBuilder buf = new StringBuilder(URL_PREFIX.length() + 20);
        buf.append(URL_PREFIX).append(getType()).append(PATH_SEPARATOR).append(getResourceName());
        ResourceUtils.addParamsToUrl(buf, getParams(), getBatchableParams());
        return buf.toString();
    }

    @Override
    public String getVersion(final WebResourceIntegration integration)
    {
        return integration.getSuperBatchVersion();
    }

    @Override
    public String getType()
    {
        return resource.suffix();
    }

    @Override
    public boolean isCacheSupported()
    {
        return true;
    }

    @Override
    public String getResourceName()
    {
        return resource.name();
    }

    @Override
    public String getModuleCompleteKey()
    {
        return "superbatch";
    }
    
    void addBatchedWebResourceDescriptor(BatchedWebResourceDescriptor descriptor)
    {
        batchedWebResourceDescriptors.add(descriptor);
    }

    @Override
    public String toString()
    {
        return "[Superbatch type=" + getType() + ", params=" + getParams() + "]";
    }
}
