package com.atlassian.plugin.webresource;

import static com.atlassian.plugin.util.EfficientStringUtils.endsWith;
import static com.atlassian.plugin.webresource.SuperBatchPluginResource.DEFAULT_RESOURCE_NAME_PREFIX;
import static com.atlassian.plugin.webresource.SuperBatchPluginResource.URL_PREFIX;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.cache.filecache.FileCache;
import com.atlassian.plugin.servlet.DownloadableResource;

import com.atlassian.plugin.webresource.cache.CacheHandle;
import com.atlassian.plugin.webresource.cache.FileCacheKey;
import com.google.common.base.Function;

import javax.annotation.Nullable;
import java.util.Map;

/**
 * Produces a batch containing all the defined super batch dependencies
 * @since 2.9.0
 */
class SuperBatchDownloadableResourceBuilder extends AbstractBatchResourceBuilder
{
    private final ResourceDependencyResolver dependencyResolver;

    public SuperBatchDownloadableResourceBuilder(final ResourceDependencyResolver dependencyResolver, final PluginAccessor pluginAccessor,
                                                 final WebResourceUrlProvider webResourceUrlProvider, final DownloadableResourceFinder resourceFinder,
                                                 final FileCache<FileCacheKey> cache, ResourceBatchingConfiguration resourceBatchingConfiguration)
    {
        super(pluginAccessor, webResourceUrlProvider, resourceFinder, cache, resourceBatchingConfiguration);
        this.dependencyResolver = dependencyResolver;
    }

    public boolean matches(final String path)
    {
        final String type = ResourceUtils.getType(path);
        return (path.indexOf(URL_PREFIX) != -1) && endsWith(path, DEFAULT_RESOURCE_NAME_PREFIX, ".", type);
    }

    public SuperBatchDownloadableResource parse(final String path, final Map<String, String> params)
    {
        final String type = ResourceUtils.getType(path);
        Iterable<DownloadableResource> resources = concat(transform(dependencyResolver.getSuperBatchDependencies(), new Function<WebResourceModuleDescriptor, Iterable<DownloadableResource>>() {
            @Override
            public Iterable<DownloadableResource> apply(@Nullable WebResourceModuleDescriptor moduleDescriptor) {
                return resolve(moduleDescriptor, type, params);
            }
        }));

        return new SuperBatchDownloadableResource(type, params, resources,
                CacheHandle.Builder.forRequest(cache, "superbatch", path, params),
                resourceBatchingConfiguration);
    }
}
