package com.atlassian.plugin.webresource;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.elements.ResourceLocation;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.transformer.TransformableResource;
import com.atlassian.plugin.webresource.transformer.UrlReadingWebResourceTransformerModuleDescriptor;
import com.atlassian.plugin.webresource.url.UrlBuilder;
import com.atlassian.plugin.webresource.transformer.WebResourceTransformerModuleDescriptor;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.Validate;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents a set of transformer invocations for a specific web resource set and extension.  Transformers are retrieved
 * from the plugin system on request, not plugin initialisation, since plugin start order is indeterminate.
 *
 * @since 2.5.0
 */
public class WebResourceTransformation
{
    private final String extension;
    private final Map<String, Element> transformerElements;
    private Logger log = LoggerFactory.getLogger(WebResourceTransformation.class);

    public WebResourceTransformation(Element element)
    {
        Validate.notNull(element.attribute("extension"));
        
        this.extension = "." + element.attributeValue("extension");

        LinkedHashMap<String, Element> transformers = new LinkedHashMap<String, Element>();
        for (Element transformElement : (List<Element>)element.elements("transformer"))
        {
            transformers.put(transformElement.attributeValue("key"), transformElement);
        }
        transformerElements = Collections.unmodifiableMap(transformers);
    }

    public boolean matches(ResourceLocation location)
    {
        String loc = location.getLocation();
        if (loc == null || "".equals(loc.trim()))
        {
            loc = location.getName();
        }
        return loc.endsWith(extension);
    }

    public void addTransformParameters(PluginAccessor pluginAccessor, UrlBuilder urlBuilder)
    {
        for (Map.Entry<String, Element> entry : transformerElements.entrySet())
        {
            for (UrlReadingWebResourceTransformerModuleDescriptor descriptor : urlReadingTransformersByKey(pluginAccessor, entry.getKey()))
            {
                descriptor.getModule().makeUrlBuilder(entry.getValue()).addToUrl(urlBuilder);
            }
        }
    }

    public DownloadableResource transformDownloadableResource(PluginAccessor pluginAccessor, DownloadableResource resource, ResourceLocation resourceLocation, String filePath, QueryParams params)
    {
        DownloadableResource lastResource = resource;
        for (Map.Entry<String, Element> entry : transformerElements.entrySet())
        {
            boolean found = false;
            for (UrlReadingWebResourceTransformerModuleDescriptor descriptor : urlReadingTransformersByKey(pluginAccessor, entry.getKey()))
            {
                found = true;
                TransformableResource transformableResource = new TransformableResource(resourceLocation, filePath, lastResource);
                lastResource = descriptor.getModule().makeResourceTransformer(entry.getValue()).transform(transformableResource, params);
            }
            if (!found)
            {
                for (WebResourceTransformerModuleDescriptor descriptor : transformersByKey(pluginAccessor, entry.getKey()))
                {
                    found = true;
                    lastResource = descriptor.getModule().transform(entry.getValue(), resourceLocation, filePath, lastResource);
                }
            }
            if (!found)
            {
                log.warn("Web resource transformer {} not found for resource {}, skipping", entry.getKey(), resourceLocation.getName());
            }
        }
        return lastResource;
    }

    private Iterable<UrlReadingWebResourceTransformerModuleDescriptor> urlReadingTransformersByKey(PluginAccessor pluginAccessor, final String key)
    {
        return transformersByKey(pluginAccessor, UrlReadingWebResourceTransformerModuleDescriptor.class, key);
    }

    private Iterable<WebResourceTransformerModuleDescriptor> transformersByKey(PluginAccessor pluginAccessor, final String key)
    {
        return transformersByKey(pluginAccessor, WebResourceTransformerModuleDescriptor.class, key);
    }

    private <D extends ModuleDescriptor<?>> Iterable<D> transformersByKey(PluginAccessor pluginAccessor, Class<D> clazz, final String key)
    {
        return Iterables.filter(pluginAccessor.getEnabledModuleDescriptorsByClass(clazz),
                new Predicate<D>()
                {
                    @Override
                    public boolean apply(@Nullable D descriptor)
                    {
                        return descriptor.getKey().equals(key);
                    }
                });
    }
}
