package com.atlassian.plugin.webresource;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.webresource.url.NamedHash;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableSet;

import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.servlet.AbstractFileServerServlet.SERVLET_PATH;

/**
 * Represents a batch of plugin resources. <p/>
 * <p/>
 * It provides methods to parse and generate urls to locate a batch of plugin resources. <p/>
 * <p/>
 * @since 2.2
 */
public class BatchPluginResource extends AbstractPluginResource
{
    /**
     * The url prefix for a batch of plugin resources: "/download/batch/"
     */
    static final String URL_PREFIX = PATH_SEPARATOR + SERVLET_PATH + PATH_SEPARATOR + "batch";

    private final ResourceKey resource;

    // not thread-safe but a safe race on setting as hash function is referentially transparent
    private int hash = 0;
    
    /**
     * This constructor should only ever be used internally within this class. It does not ensure that the resourceName's
     * file extension is the same as the given type. It is up to the calling code to ensure this.
     *
     * @param resource - the resource key
     * @param hashes - static hashes to contribute to the URL hash
     * @param params - the parameters of the resource (ieonly, media, etc)
     * @param batchableParams - batchableParams parameters of the resource
     */
    BatchPluginResource(final ResourceKey resource, final List<NamedHash> hashes, final Map<String, String> params, final Map<String, String> batchableParams, BatchedWebResourceDescriptor batchedWebResourceDescriptor)
    {
        super(hashes, params, batchableParams, ImmutableSet.of(batchedWebResourceDescriptor));
        this.resource = resource;
    }

    /**
     * Returns a url string in the format: /download/batch/MODULE_COMPLETE_KEY/resourceName?PARAMS
     * <p/>
     * e.g. /download/batch/example.plugin:webresources/example.plugin:webresources.css?ie=true
     * <p/>
     * It is important for the url structure to be:
     * 1. the same number of sectioned paths as the SinglePluginResource
     * 2. include the module complete key in the path before the resource name
     * This is due to css resources referencing other resources such as images in relative path forms.
     */
    @Override
    public String getUrl()
    {
        final StringBuilder sb = new StringBuilder();
        sb.append(URL_PREFIX).append(PATH_SEPARATOR).append(resource.key()).append(PATH_SEPARATOR).append(
            resource.name());
        ResourceUtils.addParamsToUrl(sb, params, batchableParams);
        return sb.toString();
    }

    @Override
    public String getResourceName()
    {
        return resource.name();
    }

    public String getVersion(final WebResourceIntegration integration)
    {
        final Plugin plugin = integration.getPluginAccessor().getEnabledPluginModule(
            getModuleCompleteKey()).getPlugin();
        return plugin.getPluginInformation().getVersion();
    }

    public String getModuleCompleteKey()
    {
        return resource.key();
    }

    public boolean isCacheSupported()
    {
        return !"false".equals(params.get("cache"));
    }

    @Override
    public String getType()
    {
        return resource.suffix();
    }

    @Override
    public boolean equals(final Object o)
    {
        if (this == o)
            return true;

        if ((o == null) || (getClass() != o.getClass()))
            return false;

        final BatchPluginResource that = (BatchPluginResource) o;

        return Objects.equal(params, that.params) &&
            Objects.equal(resource, that.resource);
    }

    @Override
    public int hashCode()
    {
        if (hash != 0)
            hash = Objects.hashCode(resource, params);

        return hash;
    }

    @Override
    public String toString()
    {
        return "[moduleCompleteKey=" + resource.key() + ", type=" + resource.suffix() + ", params=" + params + "]";
    }
}
