package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.url.NamedHash;
import com.atlassian.plugin.webresource.util.HashBuilder;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Abstract base class for plugin resources
 * @since v3.0
 */
public abstract class AbstractPluginResource implements PluginResource
{
    private static final Logger logger = LoggerFactory.getLogger(AbstractPluginResource.class);

    protected List<NamedHash> hashes;
    protected final Map<String, String> params;
    protected final Map<String, String> batchableParams;
    protected final Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptors;

    protected AbstractPluginResource(List<NamedHash> hashes, Map<String, String> params, Map<String, String> batchableParams,
        Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptors)
    {
        this.hashes = ImmutableList.copyOf(hashes);
        this.params = ImmutableMap.copyOf(params);
        this.batchableParams = ImmutableMap.copyOf(batchableParams);
        this.batchedWebResourceDescriptors = batchedWebResourceDescriptors;
    }

    @Override
    public List<NamedHash> getHashes()
    {
        return hashes;
    }

    @Override
    public Map<String, String> getParams()
    {
        return params;
    }

    @Override
    public Map<String, String> getBatchableParams()
    {
        return batchableParams;
    }

    @Override
    public Set<BatchedWebResourceDescriptor> getBatchedWebResourceDescriptors()
    {
        return batchedWebResourceDescriptors;
    }

    @Override
    public String getStaticUrl(WebResourceIntegration webResourceIntegration, WebResourceUrlProvider webResourceUrlProvider, UrlMode urlMode)
    {
        final String prefix;
        if (isCacheSupported())
        {
            List<String> hashes = Lists.transform(getHashes(), new Function<NamedHash, String>()
            {
                @Override
                public String apply(@Nullable NamedHash namedHash)
                {
                    return namedHash.getHash();
                }
            });
            String contributedHash = HashBuilder.buildHash(hashes);
            prefix = webResourceUrlProvider.getStaticResourcePrefix(contributedHash, getVersion(webResourceIntegration), urlMode);
        }
        else
        {
            prefix = webResourceUrlProvider.getBaseUrl(urlMode);
        }

        if (logger.isDebugEnabled())
        {
            logger.debug("Static hashes for url: %s", getUrl());
            for (NamedHash namedHash : getHashes())
            {
                logger.debug("%s: %s", namedHash.getName(), namedHash.getHash());
            }
        }
        return prefix + getUrl();
    }
}
