package com.atlassian.plugin.webresource;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.servlet.AbstractFileServerServlet.SERVLET_PATH;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;

/**
 * Represents a batch of all resources that declare themselves as part of a given context(s).
 * <p/>
 * The URL for batch resources is /download/contextbatch/&lt;type>/&lt;contextname>/batch.&lt;type. The additional type part in the path
 * is simply there to make the number of path-parts identical with other resources, so relative URLs will still work
 * in CSS files.
 * <p/>
 * The &lt;contextname&gt; part has the following syntax - <br/>
 * &lt;includedContext1&gt;,&lt;includedContext2&gt;,-&lt;excludedContext1&gt;,-&lt;excludedContext2&gt;
 * <br/>
 * To clarify, multiple comma separated contexts should be combined with any contexts preceded by a '-' to be excluded.
 *
 * @since 2.9.0
 */
class ContextBatchPluginResource implements PluginResource
{
    static final String URL_PREFIX = PATH_SEPARATOR + SERVLET_PATH + PATH_SEPARATOR + "contextbatch" + PATH_SEPARATOR;

    private final ResourceKey resource;
    private final String key;
    private final String hash;
    private final List<String> contexts;
    private final Iterable<String> excludedContexts;    
    private final Map<String, String> params;
    private final Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptors;

    ContextBatchPluginResource(final String key, List<String> contexts, final Iterable<String> excludedContexts,
            final String hash, final String type, final Map<String, String> params, Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptor)
    {
        resource = ResourceKey.Builder.batch(type);
        this.params = ImmutableMap.copyOf(params);
        this.key = key;
        this.hash = hash;
        this.contexts = contexts;
        this.excludedContexts = excludedContexts;
        this.batchedWebResourceDescriptors = Collections.unmodifiableSet(batchedWebResourceDescriptor);
    }
    
    ContextBatchPluginResource(final String key, final String hash, final String type, final Map<String, String> params, Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptor)
    {
        this(key, Collections.<String>emptyList(), Collections.<String>emptyList(), hash, type, params, batchedWebResourceDescriptor);
    }

    Iterable<String> getContexts()
    {
        return contexts;
    }

    public Iterable<String> getExcludedContexts()
    {
        return excludedContexts;
    }    
    
    public String getUrl()
    {
        final StringBuilder buf = new StringBuilder(URL_PREFIX.length() + 20);
        buf.append(URL_PREFIX).append(getType()).append(PATH_SEPARATOR).append(key).append(PATH_SEPARATOR).append(getResourceName());
        ResourceUtils.addParamsToUrl(buf, getParams());
        return buf.toString();
    }

    public Map<String, String> getParams()
    {
        return params;
    }
    
    public String getVersion(final WebResourceIntegration integration)
    {
        return hash;
    }

    public String getType()
    {
        return resource.suffix();
    }

    public boolean isCacheSupported()
    {
        return true;
    }

    public String getResourceName()
    {
        return resource.name();
    }

    public String getModuleCompleteKey()
    {
        return "contextbatch-" + resource.name();
    }
        
    @Override
    public Set<BatchedWebResourceDescriptor> getBatchedWebResourceDescriptors()
    {
        return batchedWebResourceDescriptors;
    }

    @Override
    public String toString()
    {
        return "[Context Batch type=" + getType() + ", params=" + params + "]";
    }
}
