package com.atlassian.plugin.webresource;

import com.atlassian.plugin.cache.filecache.FileCacheStreamProvider;
import com.atlassian.plugin.servlet.DownloadException;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.cache.CacheHandle;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;

import static com.google.common.collect.Iterables.any;

/**
 * A base class that batches together a sequence of DownloableResources
 *
 * @since 2.13
 */
public abstract class AbstractBatchDownloadableResource implements DownloadableResource
{
    private static final Logger log = LoggerFactory.getLogger(AbstractBatchDownloadableResource.class);
    
    private static final ResourceContentAnnotator[] DEFAULT_ANNOTATORS = new ResourceContentAnnotator[] {
        new NewlineResourceContentAnnotator()
    };
    
    private static final ResourceContentAnnotator[] JS_WRAP_ANNOTATORS = new ResourceContentAnnotator[] {
        new NewlineResourceContentAnnotator(), new TryCatchJsResourceContentAnnotator()
    };
    
    private final String type;
    private final Map<String, String> params;
    private final Iterable<DownloadableResource> resources;
    private final CacheHandle cacher;
    private final ResourceContentAnnotator[] resourceContentAnnotators;

    /**
     * This constructor should only ever be used internally within this class. It does not ensure that the resourceName's
     * file extension is the same as the given type. It is up to the calling code to ensure this.
     * @param type - the type of resource (CSS/JS)
     * @param params - the parameters of the resource (ieonly, media, etc)
     * @param resources - the resources included in the batch.
     */
    AbstractBatchDownloadableResource(final String type, final Map<String, String> params,
                                      final Iterable<DownloadableResource> resources,
                                      CacheHandle cacher)
    {
        this.type = type;
        this.params = ImmutableMap.copyOf(params);
        this.resources = resources;
        this.cacher = cacher;
        this.resourceContentAnnotators = getAnnotators(); 
    }

    /**
     * @return true if there are no resources included in this batch
     */
    public boolean isEmpty()
    {
        return Iterables.isEmpty(resources);
    }

    public boolean isResourceModified(final HttpServletRequest request, final HttpServletResponse response)
    {
        return any(resources, new Predicate<DownloadableResource>()
        {
            public boolean apply(final DownloadableResource resource)
            {
                return resource.isResourceModified(request, response);
            }
        });
    }

    public void serveResource(final HttpServletRequest request, final HttpServletResponse response) throws DownloadException
    {
        if (log.isDebugEnabled())
        {
            log.debug("Start to serve batch " + toString());
        }
        
        OutputStream out;
        try
        {
            out = response.getOutputStream();
        }
        catch (final IOException e)
        {
            throw new DownloadException(e);
        }

        streamResourceInternal(out, resourceContentAnnotators);
    }

    public void streamResource(final OutputStream originalOut) throws DownloadException
    {
        streamResourceInternal(originalOut,resourceContentAnnotators);
    }
    
    private void streamResourceInternal(final OutputStream originalOut, final ResourceContentAnnotator[] annotators) throws DownloadException
    {
        final FileCacheStreamProvider streamProvider = new FileCacheStreamProvider() {
            @Override
            public void writeStream(OutputStream dest) throws DownloadException {
                for (final DownloadableResource resource : resources)
                {
                    try
                    {
                        applyBeforeAnnotators(dest, annotators);                    
                        resource.streamResource(dest);
                        applyAfterAnnotators(dest, annotators);
                    }
                    catch (IOException ex)
                    {
                        throw new DownloadException(ex);
                    }
                }
            }
        };
        cacher.stream(originalOut, streamProvider);
    }

    public String getContentType()
    {
        final String contentType = params.get("content-type");
        if (contentType != null)
        {
            return contentType;
        }
        return null;
    }

    public Map<String, String> getParams()
    {
        return params;
    }

    public String getType()
    {
        return type;
    }
    
    private void applyBeforeAnnotators(OutputStream str, ResourceContentAnnotator[] annotators) throws IOException
    {
        for (ResourceContentAnnotator annotator : annotators)
        {
            annotator.before(str);
        }
    }
    
    /**
     * Apply the after annotators in reverse order.
     * 
     * @param str
     * @throws IOException
     */
    private void applyAfterAnnotators(OutputStream str, ResourceContentAnnotator[] annotators) throws IOException
    {
        for (int i = annotators.length - 1; i >= 0; i--)
        {
            annotators[i].after(str);
        }
    }
    
    /**
     * @return the annotators to use when writing the requested Batch resource.
     */
    private ResourceContentAnnotator[] getAnnotators() 
    {
        if ("js".equals(this.getType()))
        {
            String trycatch = System.getProperty(DefaultResourceBatchingConfiguration.PLUGIN_WEBRESOURCE_TRY_CATCH_WRAPPING);
            if (trycatch != null && Boolean.parseBoolean(trycatch))
            {
                return JS_WRAP_ANNOTATORS;
            }
            else
            {
                return DEFAULT_ANNOTATORS;
            }
        }
        else
        {
            return DEFAULT_ANNOTATORS;
        }
    }
}
