WRM.define("wrm/require-handler",
    ["wrm/jquery", "wrm/_", "wrm/builder", "wrm/conditional-comment-filter", "wrm/ie-only-filter"],
    function($, _, Builder, conditionalCommentFilter, ieOnlyFilter) {

    var WEBRESOURCE_PATTERN = /^wr!(.*)$/,
        CONTEXT_PATTERN = /^wrc!(.*)$/,
        LOADER_PATTERN = /^.+!/;

    var RequireHandler = function() {
        this._requireCache = {};

        this._requestedResources = {
            contexts: [],
            webResources: []
        };

        // Contains a promise for the currently executing to the WRM REST API, otherwise false
        this._requestInFlight = false;

        // Contains a promise for the queued request, otherwise false if there is no request currently queued.
        this._requestWaiting = false;
    };
    RequireHandler.prototype = {
        /**
         * Requires resources on the page.~
         * @param resources list of resources (eg webresources or webresource contexts)
         * @param cb Callback that is executed when the actual JS / CSS resources representing the given
         * resources have been included on the page
         * @return a Promise that is resolved when all JS / CSS resources have been included on the page. Can be used
         * as an alternative to callback.
         */
        require: function(resources, cb) {
            if (!_.isArray(resources)) {
                resources = [resources];
            }
            // Requests are cached in this._requireCache so that if a client makes multiple requests for the same
            // set of resources, we can piggyback off the promise for the first request.
            // In other words, if a client calls require([a, b]), then does some work, then calls require([a, b])
            // again, the second call should resolve immediately (or after the first call has resolved).
            if (!this._requireCache.hasOwnProperty(resources)) {
                var deferred = $.Deferred();

                function resolveAsync() {
                    this._resolveAsync(resources).then(deferred.resolve, deferred.reject);
                }

                var InOrderLoader = window.WRM && window.WRM.InOrderLoader;
                if (InOrderLoader !== undefined) {
                    InOrderLoader.registerAllResolvedCallback(resolveAsync.bind(this));
                } else {
                    resolveAsync.call(this);
                }

                this._requireCache[resources] = deferred.promise();
            }

            var promise = this._requireCache[resources];
            if (cb) {
                promise.done(function() {
                    cb.apply(this, arguments);
                });
            }
            return promise;
        },
        /**
         * Given a list of resources, translates those resources to actual CSS / JS files and includes them on the page
         * @param resources a list of webresources and webresource contexts to include
         * @return a Promise that is resolved only after all resources have been included on the page
         * @private
         */
        _resolveAsync: function(resources) {
            this._addToRequestedResources(resources);

            var deferred = $.Deferred();
            var that = this;

            // If there isn't a request currently being executed, continue as normal
            if (this._requestInFlight === false) {
                this._requestInFlight = this._getScriptsForResources().done(this._processResourceResponse.bind(this, deferred, resources));

            // There's currently a request being processed, so we need to wait for it to finish before we make the
            // next request.
            } else if (this._requestWaiting === false) {
                this._requestWaiting = deferred;
                this._requestInFlight.done(function() {
                    that._requestInFlight = that._getScriptsForResources().done(that._processResourceResponse.bind(that, deferred, resources));
                    that._requestWaiting = false;
                });

            // If _requestWaiting !== false, it means there's already a request queued up waiting to go. We've already
            // added the resources to the queued request above (this._addToRequestedResources), so we can just return the
            // promise for the queued request.
            } else {
                return this._requestWaiting.promise();
            }

            return deferred.promise();
        },
        /**
         * Processes a response from the WRM REST endpoint
         * @param resourceResponse
         * @private
         */
        _processResourceResponse: function(deferred, resources, resourceResponse) {
            this._requestInFlight = false;
            var that = this;

            if (resourceResponse.unparsedData) {
                window.WRM._unparsedData || (window.WRM._unparsedData = {});
                _.each(resourceResponse.unparsedData, function(val, key) {
                    window.WRM._unparsedData[key] = val;
                });
                window.WRM._dataArrived();
            }
            if (resourceResponse.unparsedErrors) {
                window.WRM._unparsedErrors || (window.WRM._unparsedErrors = {});
                _.each(resourceResponse.unparsedErrors, function(val, key) {
                    window.WRM._unparsedErrors[key] = val;
                });
                window.WRM._dataArrived();
            }
            var curlResources = [];
            var cssMediaResources = [];
            var filteredResourceUrlInfos = this._filter(resourceResponse.resources);
            for (var i = 0; i < filteredResourceUrlInfos.length; ++i) {
                var resource = filteredResourceUrlInfos[i];
                var url = resource.url;

                if (!this._isDarkFeatureEnabled("wrm.disable.duplicate.resource.download.fix")) {
                    this._builder.addResource(resource.key, resource.batchType, this._loadedResources);
                }

                if (resource.resourceType === "JAVASCRIPT") {
                    if (!this._isJSInInitLoad(url)) {
                        curlResources.push("js!" + url + "!order");
                    }
                }
                else if (resource.resourceType === "CSS") {
                    if (!this._isCSSInInitLoad(url)) {
                        if (resource.media && "all" !== resource.media) {
                            // HACK: this can't be loaded by curl.js. The solution is to the DOM immediately
                            // using a <link> tag. This means two things:
                            // - the callback may be called before the CSS has been loaded, resulting in a flash
                            //   of unstyled content.
                            // - it's easier to blow the IE9 stylesheet cap (curl.js works around this).
                            cssMediaResources.push(resource);
                        }
                        else {
                            curlResources.push("css!" + url);
                        }
                    }
                }
                else {
                    AJS.log("Unknown resource type required: " + url);
                }
            }
            AJS.log("Downloading resources:\n" + curlResources.join("\n"));
            WRM.curl(curlResources, function() {
                // Add all css media resources. This is done after curl resources to ensure ordering is consistent
                // with the way resources are delivered on the server.
                _.each(cssMediaResources, function(resource) {
                    that._loadCssImmediate(resource);
                });
                var callbackArgs = _.map(resources, function() { return window; });
                deferred.resolveWith(that, callbackArgs);
            }, function() {
                deferred.rejectWith(that, arguments);
            });
        },
        /**
         * Loads
         * @param resource
         * @private
         */
        _loadCssImmediate: function(resource) {
            AJS.log('WARN: asynchronously loading a CSS resource containing a media query: ' + resource.url);
            var tag = '<link rel="stylesheet" type="text/css" href="' + resource.url + '" media="' + resource.media + '" />';
            $(tag).appendTo('head');
        },
        /**
         * Makes an AJAX request to retrieve the actual JS and CSS files required to represent a resource
         * @return a Promise for the AJAX request
         * @private
         */
        _getScriptsForResources: function() {
            if (!this._builder) {
                this._builder = new Builder();
                this._loadedResources = this._builder.initialize(document);
            }

            var promise = $.ajax({
                url: AJS.contextPath() + "/rest/webResources/1.0/resources",
                type: "POST",
                contentType: "application/json",
                dataType: "json",
                data: JSON.stringify({
                    r: this._requestedResources.webResources,
                    c: this._requestedResources.contexts,
                    xc: this._loadedResources.contexts,
                    xr: this._loadedResources.modules
                })
            });

            this._requestedResources.webResources = [];
            this._requestedResources.contexts = [];

            return promise;
        },
        /**
         *
         * @param resources list of resources (e.g. context or web resource)
         * @private
         */
        _addToRequestedResources: function(resources) {
            var that = this;

            _.each(resources, function(resource) {
                var match;
                if (match = resource.match(WEBRESOURCE_PATTERN)) {
                    that._requestedResources.webResources.push(match[1]);
                }
                else if (match = resource.match(CONTEXT_PATTERN)) {
                    that._requestedResources.contexts.push(match[1]);
                }
                // If it's neither context nor web-resource it should be AMD module.
                else {
                    // Handling modules as if it's the web-resource.
                    that._requestedResources.webResources.push(resource.replace(LOADER_PATTERN, ''));
                }
            });
        },
        /**
         * Checks if a script element whose src is the given url exists on the page
         * @param url url
         * @return {boolean} True if the script is on the page, otherwise false
         * @private
         */
        _isJSInInitLoad: function(url) {
            return $("script[src='" + url + "']").length > 0;
        },
        /**
         * Checks if a link element whose href is the given url exists on the page
         * @param url url
         * @return {boolean} True if the link is on the page, otherwise false
         * @private
         */
        _isCSSInInitLoad: function(url) {
            return $("link[href='" + url + "']").length > 0;
        },
        /**
         * Filters resources that should not be included from the given list
         * @param resourceUrlInfos list
         * @return list of filters resourceUrlInfo objects.
         * @private
         */
        _filter: function(resourceUrlInfos) {
            if (!this._filters) {
                this._filters = [
                    conditionalCommentFilter,
                    ieOnlyFilter
                ];
            }
            var filteredResourceUrlInfos = resourceUrlInfos;
            _.each(this._filters, function(filter) {
                filteredResourceUrlInfos = filter(filteredResourceUrlInfos);
            });
            return filteredResourceUrlInfos;
        },

        _isDarkFeatureEnabled: function(featureKey) {
            // Make sure the dark features API is available before we continue.
            if (_.isUndefined(AJS) ||
                _.isUndefined(AJS.DarkFeatures) ||
                _.isUndefined(AJS.DarkFeatures.isEnabled)) {
                return false;
            }

            return AJS.DarkFeatures.isEnabled(featureKey);

        }
    };

    return RequireHandler;
});
