package com.atlassian.webresource.plugin.prebake.discovery;

import com.atlassian.plugin.webresource.cdn.mapper.DefaultMapping;
import com.atlassian.plugin.webresource.cdn.mapper.DefaultMappingSet;
import com.atlassian.plugin.webresource.cdn.mapper.Mapping;
import com.atlassian.plugin.webresource.cdn.mapper.MappingParser;
import com.atlassian.plugin.webresource.impl.PrebakeErrorFactory;
import com.atlassian.webresource.api.assembler.resource.PrebakeError;
import com.atlassian.webresource.plugin.prebake.exception.PreBakeIOException;
import com.atlassian.webresource.plugin.prebake.resources.Resource;
import com.atlassian.webresource.plugin.prebake.resources.ResourceCollector;
import com.atlassian.webresource.plugin.prebake.resources.ResourceContent;
import com.atlassian.webresource.plugin.prebake.util.FileUtil;
import com.atlassian.webresource.plugin.prebake.util.PreBakeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.BUNDLE_ZIP_DIR;
import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.PREBAKE_REPORT;
import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.RELATIVE_MAPPINGS;
import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.RELATIVE_RESOURCES;
import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.RELATIVE_STATE;
import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.RELATIVE_ZIP;
import static com.atlassian.webresource.plugin.prebake.util.PreBakeUtil.SEPRTR;

import static java.lang.String.format;

/**
 * Responsible for pre bake processing.
 *
 * @since v3.5.0
 */
public class DiscoveryTask implements Runnable {

    private static final Logger log = LoggerFactory.getLogger(DiscoveryTask.class);

    private static final DecimalFormat PERCENT_FORMAT = new DecimalFormat("#0.00");

    private final WebResourcePreBaker preBaker;
    private final WebResourceBatch batch;
    private final List<WebResourceCrawler> crawlers;
    private final ResourceCollector resourceCollector;

    public DiscoveryTask(
            WebResourcePreBaker preBaker,
            WebResourceBatch batch,
            List<WebResourceCrawler> crawlers,
            ResourceCollector resourceCollector) {
        this.preBaker = preBaker;
        this.batch = batch;
        this.crawlers = crawlers;
        this.resourceCollector = resourceCollector;
    }

    private void doRun() throws Exception {
        // Clear previous results.
        try {
            Path path = Paths.get(preBaker.getBundleDir().getPath(), batch.getVersion());
            FileUtils.deleteDirectory(path.toFile());
        } catch (IOException e) {
            throw new PreBakeIOException("Problem during clearing previous results", e);
        }

        // Create new directories.
        String resourcesDir = preBaker.getBundleDir().getPath() + SEPRTR + batch.getVersion() + SEPRTR;
        String bundleZip = resourcesDir + RELATIVE_ZIP;
        String bundleDir = resourcesDir + BUNDLE_ZIP_DIR;
        String relativeState = bundleDir + RELATIVE_STATE;
        String relativeMappings = bundleDir + RELATIVE_MAPPINGS;
        String relativeResources = bundleDir + RELATIVE_RESOURCES;
        String prebakeReport = bundleDir + PREBAKE_REPORT;
        if (Paths.get(relativeResources).toFile().mkdirs()) {
            log.trace("Creating new directory: {}", relativeResources);
        }

        // Discover web resources.
        Set<Resource> resources = new LinkedHashSet<>();
        if (CollectionUtils.isNotEmpty(crawlers)) {
            crawlers.forEach(c -> resources.addAll(c.get()));
        }

        CrawlerResult cr = collectAll(resources, relativeResources);
        batch.addCrawlerResult(cr);

        // Provide bundle.zip.
        try {
            FileUtil.write2File(
                    batch.getVersion().getBytes(PreBakeUtil.UTF8),
                    new File(relativeState));
            FileUtil.write2File(
                    new MappingParser().getAsString(batch.getMappings()).getBytes(PreBakeUtil.UTF8),
                    new File(relativeMappings));
            writeBatchReport(batch, prebakeReport);
            Set<String> sources = StreamSupport.stream(batch.getMappings().all().spliterator(), false)
                    .map(Mapping::mappedResources)
                    .flatMap(Collection::stream)
                    .map(h -> relativeResources + h)
                    .sorted()
                    .collect(Collectors.toSet());
            sources.add(relativeState);
            sources.add(relativeMappings);
            sources.add(prebakeReport);
            FileUtil.zipFiles(new File(resourcesDir), sources, bundleZip);
        } catch (IOException e) {
            throw new PreBakeIOException(String.format("Problem during %s creation", bundleZip), e);
        }
    }

    private CrawlerResult collectAll(Set<Resource> resources, String resourcesDir) {
        Set<Mapping> mappings = new LinkedHashSet<>();
        List<TaintedResource> taintedResources = new ArrayList<>();

        for (Resource res : resources) {
            collect(res, resourcesDir, mappings, taintedResources);
        }

        return new CrawlerResult(new DefaultMappingSet(mappings), taintedResources);
    }

    private void collect(
            Resource res,
            String resourcesDir,
            Set<Mapping> mappings,
            List<TaintedResource> taintedResources) {
        if (!preBaker.isCSSPrebakingEnabled() && res.getExtension().equals(".css")) {
            return;
        }

        String url = res.getUrl();
        String fullName = res.getName();
        // Relative URL, used as a key for the resource -> hash mapping.
        String relativeUrl = preBaker.relativeUrl(url);
        // Complete URI, used to retrieve the resource from the WebResourceManager
        URI uri = preBaker.toURI(url);

        if (res.isTainted()) {
            log.warn(format("Encountered tainted resource: %s", url));
            taintedResources.add(new TaintedResource(url, fullName, res.getPrebakeErrors()));
            return;
        }

        try {
            ResourceContent rc = resourceCollector.collect(uri);
            byte[] content = rc.getContent();
            String fileName = "/" + PreBakeUtil.hash(content) + res.getExtension();
            Path target = Paths.get(resourcesDir, fileName);
            if (!Files.exists(target)) {
                InputStream is = new ByteArrayInputStream(content);
                Files.copy(is, target);
                is.close();
            }
            Mapping mapping = new DefaultMapping(relativeUrl, Stream.of(fileName));
            mappings.add(mapping); // Phase 1 only single hash values
        } catch (PreBakeIOException | IOException e) {
            String msg = format("Could not fetch resource: %s", url);
            log.error(msg, e);
            PrebakeError pe = PrebakeErrorFactory.from(msg, e);
            taintedResources.add(new TaintedResource(url, fullName, pe));
        }
    }

    private void writeBatchReport(WebResourceBatch batch, String filename) throws IOException {
        FileWriter fw = new FileWriter(filename);
        BufferedWriter bw = new BufferedWriter(fw);

        List<TaintedResource> trs = batch.getTaintedResources();
        int mapCount = batch.getMappings().size();
        int taintCount = trs.size();
        int total = mapCount + taintCount;
        double coverage = total == 0 ? 0 : ((double) mapCount / total) * 100;

        bw.append("Coverage: ").append(PERCENT_FORMAT.format(coverage)).append("%\n");
        bw.append("Baked resources: ").append(String.valueOf(mapCount)).append("\n");
        bw.append("Tainted resources: ").append(String.valueOf(taintCount)).append("\n");
        for (TaintedResource tr : trs) {
            bw.append("\nRESOURCE: ");
            bw.append(tr.getUrl());
            bw.append("\n");
            int i = 0;
            for (PrebakeError pe : tr.getPrebakeErrors()) {
                bw.append(String.valueOf(i)).append(") ");
                bw.append(pe.toString());
                bw.append("\n");
            }
        }

        bw.close();
        fw.close();
    }

    @Override
    public final void run() {
        preBaker.setState(PreBakeState.RUNNING);
        boolean cancelled = false;
        try {
            doRun();
        } catch (Exception e) {
            cancelled = true;
            throw new RuntimeException(e); // wrap into PreBakeException
        } finally {
            preBaker.setState(cancelled ? PreBakeState.CANCELLED : PreBakeState.DONE);
        }
    }

}
