package com.atlassian.webresource.plugin.async;

import com.atlassian.webresource.api.UrlMode;
import com.atlassian.webresource.api.assembler.WebResource;
import com.atlassian.webresource.api.assembler.WebResourceAssembler;
import com.atlassian.webresource.api.assembler.WebResourceAssemblerFactory;
import com.atlassian.webresource.api.assembler.WebResourceSet;
import com.atlassian.webresource.api.assembler.resource.PluginCssResource;
import com.atlassian.webresource.api.assembler.resource.PluginJsResource;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Set;

/**
 * Implementation of AsyncWebResourceLoader
 *
 * @since v3.0
 */
public class AsyncWebResourceLoaderImpl implements AsyncWebResourceLoader
{
    private static final Logger logger = LoggerFactory.getLogger(AsyncWebResourceLoaderImpl.class);

    private final WebResourceAssemblerFactory webResourceAssemblerFactory;

    public AsyncWebResourceLoaderImpl(WebResourceAssemblerFactory webResourceAssemblerFactory)
    {
        this.webResourceAssemblerFactory = webResourceAssemblerFactory;
    }

    @Override
    public Iterable<Resource> resolve(Set<String> webResources, Set<String> contexts, Set<String> excludeResources,
                                       Set<String> excludeContexts)
    {
        WebResourceSet webResourceSet = resolveWebResourceSet(webResources, contexts, excludeResources, excludeContexts);

        final Collection<Resource> resources = Lists.newArrayList();
        for (WebResource webResource : webResourceSet.getResources())
        {
            if (webResource instanceof PluginJsResource)
            {
                PluginJsResource jsResource = (PluginJsResource) webResource;
                resources.add(new Resource(jsResource.getStaticUrl(UrlMode.RELATIVE), ResourceType.JAVASCRIPT,
                        jsResource.getParams().conditionalComment(), jsResource.getParams().ieOnly()));
            }
            else if (webResource instanceof PluginCssResource)
            {
                // Note we don't support CSS media queries for async web resources
                PluginCssResource cssResource = (PluginCssResource) webResource;
                resources.add(new Resource(cssResource.getStaticUrl(UrlMode.RELATIVE), ResourceType.CSS,
                        cssResource.getParams().conditionalComment(), cssResource.getParams().ieOnly()));
            }
        }
        return resources;
    }

    private WebResourceSet resolveWebResourceSet(Set<String> webResources, Set<String> contexts, Set<String> excludeResources,
                                                 Set<String> excludeContexts)
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Requiring webresources {}",  webResources);
            logger.debug("Requiring contexts {}", contexts);
            logger.debug("Excluding webresource {}", excludeResources);
            logger.debug("Excluding contexts {}", excludeContexts);
        }

        WebResourceAssembler assembler = webResourceAssemblerFactory.create().build();
        assembler.resources().exclude(excludeResources, excludeContexts);

        for (String webResource : webResources)
            assembler.resources().requireWebResource(webResource);

        for (String context : contexts)
            assembler.resources().requireContext(context);

        return assembler.assembled().drainIncludedResources();
    }
}
