package com.atlassian.plugin.servlet.cache.model;

import com.google.common.annotations.VisibleForTesting;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.time.LocalDateTime;
import java.util.Objects;
import java.util.function.Predicate;

import static com.google.common.net.HttpHeaders.IF_MODIFIED_SINCE;
import static com.google.common.net.HttpHeaders.IF_NONE_MATCH;
import static java.time.ZoneOffset.UTC;
import static java.util.Objects.isNull;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.builder.ToStringBuilder.reflectionToString;

/**
 * Represents a caching information for a current {@link HttpServletRequest} headers and the {@link HttpServletResponse} response body.
 *
 * @since 4.1.18
 */
public class CacheInformation {

    /**
     * Value that represents an empty header.
     */
    @VisibleForTesting
    static final long EMPTY_MODIFIED_SINCE_HEADER = -1;

    /**
     * The HTTP {@code If-Modified-Since} header field content.
     */
    private final long ifModifiedSinceHeader;

    /**
     * The HTTP {@code If-None-Match} header field content.
     */
    private final String ifNoneMatchHeader;

    @VisibleForTesting
    public CacheInformation(final long ifModifiedSinceHeader, final String ifNoneMatchHeader) {
        this.ifModifiedSinceHeader = ifModifiedSinceHeader;
        this.ifNoneMatchHeader = ifNoneMatchHeader;
    }

    /**
     * Build a {@link CacheInformation} based on the current request headers and response body.
     *
     * @param request             The current request with the headers to be used for the caching verifications.
     */
    public CacheInformation(@Nonnull final HttpServletRequest request) {
        requireNonNull(request, "The request is mandatory to build the caching information.");
        this.ifModifiedSinceHeader = request.getDateHeader(IF_MODIFIED_SINCE);
        this.ifNoneMatchHeader = request.getHeader(IF_NONE_MATCH);
    }

    @Override
    public boolean equals(final Object other) {
        if (other instanceof CacheInformation) {
            final CacheInformation otherCachingInformation = (CacheInformation) other;
            return ifModifiedSinceHeader == otherCachingInformation.ifModifiedSinceHeader &&
                    Objects.equals(ifNoneMatchHeader, otherCachingInformation.ifNoneMatchHeader);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(ifModifiedSinceHeader, ifNoneMatchHeader);
    }

    /**
     * <p>Identify if the content of a {@link HttpServletResponse} is fresh or not.</p>
     * <p>The content is considered fresh when one of the following {@link Predicate} is fulfilled:</p>
     * <ul>
     *      <li>(The If-Modified-Since header is <b>LOWER THAN</b> the defined).</li>
     * </ul>
     *
     * @param pluginLastModifiedDate The representation of the last time that the plugin was modified.
     */
    public boolean hasFreshContent(@Nonnull final LocalDateTime pluginLastModifiedDate) {
        final long pluginLastModifiedSeconds = pluginLastModifiedDate.toEpochSecond(UTC);
        return EMPTY_MODIFIED_SINCE_HEADER == ifModifiedSinceHeader || pluginLastModifiedSeconds > ifModifiedSinceHeader;
    }

    /**
     * @see CacheInformation#hasFreshContent(LocalDateTime).
     */
    public boolean hasNotFreshContent(@Nonnull final LocalDateTime pluginLastModifiedDate) {
        return !hasFreshContent(pluginLastModifiedDate);
    }

    /**
     * @see CacheInformation#hasFreshContent(CacheableResponse).
     */
    public boolean hasNotFreshContent(@Nonnull final CacheableResponse httpServletResponse) {
        return !hasFreshContent(httpServletResponse);
    }

    /**
     * <p>Identify if the content of a {@link HttpServletResponse} is fresh or not.</p>
     * <p>The content is considered fresh when one of the following {@link Predicate} is fulfilled:</p>
     * <ul>
     *      <li>(There is an ETag Information) <b>AND</b> (The request ETag <b>DOES NOT MATCH</b> the current response body).</li>
     * </ul>
     *
     * @param httpServletResponse The response owner of the content to be compared with the provided 'If-None-Match' header.
     */
    public boolean hasFreshContent(@Nonnull final CacheableResponse httpServletResponse) {
        final String eTagToken = httpServletResponse.toETagToken()
                .map(ETagToken::getValue)
                .orElse(null);
        return isNull(eTagToken) || isNull(ifNoneMatchHeader) || !eTagToken.equals(ifNoneMatchHeader);
    }

    @Override
    public String toString() {
        return reflectionToString(this);
    }
}
