package com.atlassian.plugins.osgi.javaconfig;

import com.atlassian.annotations.PublicApi;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.util.Arrays.stream;
import static java.util.Objects.requireNonNull;

/**
 * The user-provided options for exporting a local Spring bean as an OSGi Service.
 *
 * @since 0.3
 */
@PublicApi
public class ExportOptions {

    /**
     * Factory method for an instance that exports an object under one or more service interfaces. Note that in OSGi,
     * the term "service interfaces" includes concrete and abstract classes, as well as actual interfaces. There is no
     * significance to the order in which the arguments are passed to this method.
     *
     * @param firstInterface the first service interface under which to export the object
     * @param otherInterfaces any other service interfaces under which to export the object
     * @return a new instance
     */
    public static ExportOptions as(final Class<?> firstInterface, final Class<?>... otherInterfaces) {
        return new ExportOptions(firstInterface, otherInterfaces);
    }

    private final List<Class<?>> serviceInterfaces = new ArrayList<>();
    private final Map<String, Object> properties = new HashMap<>();

    private ExportOptions(final Class<?> firstInterface, final Class<?>... otherInterfaces) {
        addServiceInterface(firstInterface);
        stream(otherInterfaces).forEach(this::addServiceInterface);
    }

    private void addServiceInterface(final Class<?> serviceInterface) {
        this.serviceInterfaces.add(requireNonNull(serviceInterface));
    }

    /**
     * Adds the given property to this service's map of OSGi properties. Other bundles can filter on these properties.
     *
     * @param key the name of the property
     * @param value the value of the property
     * @return this instance
     */
    public ExportOptions withProperty(final String key, final Object value) {
        properties.put(key, value);
        return this;
    }

    /**
     * Returns the service interfaces under which the object is to be exported to OSGi. Note that in OSGi, the term
     * "service interfaces" can include concrete and abstract classes.
     *
     * @return a non-null array
     */
    public Class<?>[] getInterfaces() {
        return new ArrayList<>(serviceInterfaces).toArray(new Class<?>[0]);
    }

    /**
     * Returns the properties to be exported for this service.
     *
     * @return a non-null map
     */
    public Map<String, Object> getProperties() {
        return new HashMap<>(properties);
    }
}
