package com.atlassian.plugins.osgi.javaconfig.moduletypes;

import com.atlassian.annotations.PublicApi;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.osgi.external.ListableModuleDescriptorFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.annotation.Nullable;
import java.util.Set;

import static java.util.Collections.singleton;
import static java.util.Objects.requireNonNull;
import static org.springframework.beans.factory.config.AutowireCapableBeanFactory.AUTOWIRE_CONSTRUCTOR;

/**
 * This class is helpful when defining your own plugin points (i.e. module types). It allows your
 * {@link ListableModuleDescriptorFactory} to remain unaware of the constructor arguments of your
 * {@link ModuleDescriptor} class. These will instead be satisfied by beans found in your plugin's Spring
 * {@link ApplicationContext}, using {@link AutowireCapableBeanFactory#AUTOWIRE_CONSTRUCTOR} mode. It is your
 * responsibility to ensure that these beans are available. For example, subclasses of
 * {@link com.atlassian.plugin.descriptors.AbstractModuleDescriptor} require the plugin system's
 * {@link com.atlassian.plugin.module.ModuleFactory} to be injected. An easy way to achieve this is to
 * {@link org.springframework.context.annotation.Import} this library's
 * {@link com.atlassian.plugins.osgi.javaconfig.configs.beans.ModuleFactoryBean} configuration class into your plugin's
 * own configuration class.
 *
 * @param <MD> the type of ModuleDescriptor
 */
@PublicApi
public abstract class PluginContextModuleDescriptorFactory<MD extends ModuleDescriptor>
        implements ListableModuleDescriptorFactory, ApplicationContextAware {

    private final Class<MD> moduleDescriptorClass;
    private final String xmlElementType;

    private ApplicationContext applicationContext;

    /**
     * Constructor
     *
     * @param xmlElementType the XML element name for this type of module in the plugin descriptor
     * @param moduleDescriptorClass the module descriptor class
     */
    protected PluginContextModuleDescriptorFactory(final String xmlElementType, final Class<MD> moduleDescriptorClass) {
        this.moduleDescriptorClass = requireNonNull(moduleDescriptorClass, "moduleDescriptorClass cannot be null");
        this.xmlElementType = requireNonNull(xmlElementType, "xmlElementType cannot be null");
    }

    @Override
    public void setApplicationContext(final ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = requireNonNull(applicationContext);
    }

    @Nullable
    @Override
    public ModuleDescriptor getModuleDescriptor(final String type) throws PluginParseException {
        if (this.xmlElementType.equals(type)) {
            // This bean creation logic was copied from SpringHostContainer
            final AutowireCapableBeanFactory autowireCapableBeanFactory = applicationContext.getAutowireCapableBeanFactory();
            final Object moduleDescriptor = autowireCapableBeanFactory.createBean(
                    moduleDescriptorClass, AUTOWIRE_CONSTRUCTOR, false);
            //noinspection unchecked
            return (MD) moduleDescriptor;
        }
        return null;
    }

    @Override
    public boolean hasModuleDescriptor(final String type) {
        return (this.xmlElementType.equals(type));
    }

    @Nullable
    @Override
    public Class<? extends ModuleDescriptor> getModuleDescriptorClass(final String type) {
        return (this.xmlElementType.equals(type) ? moduleDescriptorClass : null);
    }

    @Override
    public Iterable<String> getModuleDescriptorKeys() {
        return singleton(xmlElementType);
    }

    @Override
    public Set<Class<? extends ModuleDescriptor>> getModuleDescriptorClasses() {
        return singleton(moduleDescriptorClass);
    }
}
