package com.atlassian.plugins.osgi.javaconfig;

import com.atlassian.annotations.PublicApi;
import com.atlassian.plugin.osgi.external.ListableModuleDescriptorFactory;
import org.eclipse.gemini.blueprint.context.BundleContextAware;
import org.eclipse.gemini.blueprint.service.exporter.OsgiServiceRegistrationListener;
import org.eclipse.gemini.blueprint.service.exporter.support.DefaultInterfaceDetector;
import org.eclipse.gemini.blueprint.service.exporter.support.ExportContextClassLoaderEnum;
import org.eclipse.gemini.blueprint.service.exporter.support.OsgiServiceFactoryBean;
import org.eclipse.gemini.blueprint.service.importer.support.OsgiServiceProxyFactoryBean;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceRegistration;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Nullable;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

import static java.util.Collections.emptyMap;
import static java.util.stream.Stream.concat;

/**
 * This class is the main entry point for this library. It provides utility methods for creating the Spring beans
 * necessary to import/export OSGi services. This approach is a programmatic alternative to using Atlassian Spring
 * Scanner, which creates similar bean definitions at runtime based on text files bundled into the plugin at build time.
 *
 * @since 0.1
 */
@PublicApi
public final class OsgiServices {

    /**
     * Returns a factory for a Spring bean that looks up the given OSGi service. You would typically use this method
     * when the given service class is not guaranteed to be on the classpath at runtime, e.g. for cross-product plugins,
     * a service that only exists in one product. In other cases, it's simpler to call {@link #importOsgiService(Class)}.
     *
     * Typically, you would call this method from the body of an <code>@Bean</code> method in a Spring Java
     * <code>@Configuration</code> class, for example:
     *
     * <pre>
     *     &#064;Configuration
     *     public class MyJavaBeans {
     *
     *         &#064;Bean
     *         public FactoryBean&lt;FooService&gt; fooService() {
     *             return factoryBeanForOsgiService(FooService.class);
     *         }
     *     }
     * </pre>
     *
     * This effectively adds a <code>FooService</code> bean to your Spring <code>ApplicationContext</code> (Spring
     * transparently handles the invocation of the <code>FactoryBean</code> as necessary).
     *
     * Because the generic type of the {@link FactoryBean} is erased at runtime, if the <code>FooService</code>
     * class is not actually available, you won't get a {@link NoClassDefFoundError} unless you actually invoke
     * the FactoryBean. If necessary, you can prevent this happening via an appropriate bean <code>Condition</code>
     * (for example, that checks which product is hosting the plugin).
     *
     * This method is an alternative to putting a <code>&lt;component-import&gt;</code> element into your plugin XML or
     * using Spring Scanner's <code>@ComponentImport</code> annotation.
     *
     * @param serviceInterface the type of service to obtain from OSGi (must be an interface)
     * @param <T> the type of service
     * @return a FactoryBean that creates a proxy for a service obtained from OSGi
     */
    @SuppressWarnings("unused")
    public static <T> FactoryBean<T> factoryBeanForOsgiService(final Class<T> serviceInterface) {
        return factoryBeanForOsgiService(serviceInterface, null);
    }

    /**
     * Returns a factory for a Spring bean that looks up the given OSGi service. You would typically use this method
     * when the given service class is not guaranteed to be on the classpath at runtime, e.g. for cross-product plugins,
     * a service that only exists in one product. In other cases, it's simpler to call {@link #importOsgiService(Class)}.
     *
     * Typically, you would call this method from the body of an <code>@Bean</code> method in a Spring Java
     * <code>@Configuration</code> class, for example:
     *
     * <pre>
     *     &#064;Configuration
     *     public class MyJavaBeans {
     *
     *         &#064;Bean
     *         public FactoryBean&lt;FooService&gt; fooService() {
     *             return factoryBeanForOsgiService(FooService.class);
     *         }
     *     }
     * </pre>
     *
     * This effectively adds a <code>FooService</code> bean to your Spring <code>ApplicationContext</code> (Spring
     * transparently handles the invocation of the <code>FactoryBean</code> as necessary).
     *
     * Because the generic type of the {@link FactoryBean} is erased at runtime, if the <code>FooService</code>
     * class is not actually available, you won't get a {@link NoClassDefFoundError} unless you actually invoke
     * the FactoryBean. If necessary, you can prevent this happening via an appropriate bean <code>Condition</code>
     * (for example, that checks which product is hosting the plugin).
     *
     * This method is an alternative to putting a <code>&lt;component-import&gt;</code> element into your plugin XML or
     * using Spring Scanner's <code>@ComponentImport</code> annotation.
     *
     * @param serviceInterface the type of service to obtain from OSGi (must be an interface)
     * @param filter an optional filter to apply to the service lookup, e.g. {@code (foo=bar)} means the service will
     *               only be imported if its {@code foo} property is set to {@code bar}.
     * @param <T> the type of service
     * @return a FactoryBean that creates a proxy for a service obtained from OSGi
     */
    @SuppressWarnings({"unchecked", "WeakerAccess"})
    public static <T> FactoryBean<T> factoryBeanForOsgiService(
            final Class<T> serviceInterface, @Nullable final String filter) {
        final OsgiServiceProxyFactoryBean factoryBean = new OsgiServiceProxyFactoryBean();
        factoryBean.setFilter(filter);
        factoryBean.setInterfaces(new Class[] {serviceInterface});
        factoryBean.setBeanClassLoader(serviceInterface.getClassLoader());
        return (FactoryBean<T>) factoryBean;
    }

    /**
     * Obtains an instance of the given service from OSGi. This is the simplest
     * way to make an OSGi service (technically, its proxy) available as a Spring
     * bean, but it only works in cases where the {@code serviceClass} is
     * guaranteed to be on the classpath at runtime (e.g. a single-product
     * plugin, or a cross-product plugin where it's a platform service).
     *
     * Typically, you would call this method from the body of an <code>@Bean</code> method in a Spring Java
     * <code>@Configuration</code> class, for example:
     *
     * <pre>
     *     &#064;Configuration
     *     public class MyJavaBeans {
     *
     *         &#064;Bean
     *         public FooService fooService() {
     *             return importOsgiService(FooService.class);
     *         }
     *     }
     * </pre>
     *
     * This is a code-only alternative to putting a <code>&lt;component-import&gt;</code> element into your plugin XML
     * or using Spring Scanner's {@code @ComponentImport} annotation.
     *
     * @param serviceClass the type of service to obtain
     * @param <T> the type of service
     * @return the service
     */
    public static <T> T importOsgiService(final Class<T> serviceClass) {
        return importOsgiService(serviceClass, null);
    }

    /**
     * Obtains an instance of the given service from OSGi. This is the simplest
     * way to make an OSGi service (technically, its proxy) available as a Spring
     * bean, but it only works in cases where the {@code serviceClass} is
     * guaranteed to be on the classpath at runtime (e.g. a single-product
     * plugin, or a cross-product plugin where it's a platform service).
     *
     * The {@code filter} parameter allows you to import only a service whose properties match that filter.
     *
     * Typically, you would call this method from the body of an <code>@Bean</code> method in a Spring Java
     * <code>@Configuration</code> class, for example:
     *
     * <pre>
     *     &#064;Configuration
     *     public class MyJavaBeans {
     *
     *         &#064;Bean
     *         public FooService fooService() {
     *             String propertyFilter = "(language=en)";
     *             return importOsgiService(FooService.class, propertyFilter);
     *         }
     *     }
     * </pre>
     *
     * This is a code-only alternative to putting a <code>&lt;component-import&gt;</code> element into your plugin XML
     * or using Spring Scanner's {@code @ComponentImport} annotation.
     *
     * @param serviceClass the type of service to obtain
     * @param filter an optional filter to apply to the service lookup, e.g. {@code (foo=bar)} means the service will
     *               only be imported if its {@code foo} property is set to {@code bar}.
     * @param <T> the type of service
     * @return the service
     */
    public static <T> T importOsgiService(final Class<T> serviceClass, @Nullable final String filter) {
        final FactoryBean<T> factoryBean = factoryBeanForOsgiService(serviceClass, filter);
        try {
            if (factoryBean instanceof BundleContextAware) {
                final BundleContext bundleContext = FrameworkUtil.getBundle(OsgiServices.class).getBundleContext();
                ((BundleContextAware) factoryBean).setBundleContext(bundleContext);
            }
            if (factoryBean instanceof InitializingBean) {
                ((InitializingBean) factoryBean).afterPropertiesSet();
            }

            return factoryBean.getObject();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Defines a new type of plugin module (i.e. a new plugin point).
     *
     * This is a code-only alternative to Spring Scanner's {@code @ModuleType} annotation.
     *
     * @param moduleDescriptorFactory the module descriptor factory for the new module type
     * @return a factory bean for the service registration
     */
    public static FactoryBean<ServiceRegistration> exportAsModuleType(
            final ListableModuleDescriptorFactory moduleDescriptorFactory) {
        // We define the module type by exporting the given module descriptor factory as an OSGi service; this technique
        // is described at https://developer.atlassian.com/server/framework/atlassian-sdk/module-type-plugin-module/
        return exportOsgiService(moduleDescriptorFactory, emptyMap(), ListableModuleDescriptorFactory.class);
    }

    /**
     * Exports the given bean as an OSGi service, under one or more interfaces.
     *
     * This is a code-only alternative to either marking a <code>&lt;component&gt;</code> element with
     * <code>public="true"</code> in your plugin XML or using Spring Scanner's {@code @ExportAsService} annotation.
     * A difference in behaviour is that this method publishes the component only via the given interfaces, whereas
     * Spring Scanner defaults to publishing the component under all the interfaces it happens to implement (which is
     * often not what you want).
     *
     * @param bean the bean to export
     * @param serviceProps any service properties you wish to define (pass null or an empty map for none)
     * @param firstInterface the first interface under which to publish the bean
     * @param otherInterfaces any other interfaces under which to publish the bean; there is no
     *                        hierarchy between these and the "first" interface; this API is
     *                        simply to enforce the provision of at least one interface
     * @return a factory bean for the service registration
     */
    public static FactoryBean<ServiceRegistration> exportOsgiService(
            final Object bean, @Nullable final Map<String, Object> serviceProps, final Class<?> firstInterface,
            final Class<?>... otherInterfaces) {
        final Map<String, Object> mutableServiceProps = new HashMap<>(serviceProps == null ? emptyMap() : serviceProps);

        final Class[] interfaces = concatClasses(firstInterface, otherInterfaces);

        final OsgiServiceFactoryBean exporter = new OsgiServiceFactoryBean();
        exporter.setInterfaceDetector(DefaultInterfaceDetector.DISABLED);
        exporter.setBeanClassLoader(bean.getClass().getClassLoader());
        exporter.setExportContextClassLoader(ExportContextClassLoaderEnum.UNMANAGED);
        exporter.setInterfaces(interfaces);
        exporter.setServiceProperties(mutableServiceProps);
        exporter.setTarget(bean);
        exporter.setListeners(new OsgiServiceRegistrationListener[0]);
        return exporter;
    }

    private static Class<?>[] concatClasses(final Class<?> firstInterface, final Class<?>[] otherInterfaces) {
        return concat(Stream.of(firstInterface), Stream.of(otherInterfaces))
                    .toArray(Class[]::new);
    }

    private OsgiServices() {
    }
}
