package com.atlassian.plugin.util;

import org.apache.commons.lang3.StringUtils;

import java.math.BigInteger;
import java.util.Comparator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Compares dotted version strings of varying length. Makes a best effort with
 * other delimiters and non-numeric versions.
 * <p>
 * For dotted decimals, comparison is as you'd expect: 0.1 is before 0.2 is before
 * 1.0 is before 2.0. This works for any number of dots.
 * <p>
 * More complicated version numbers are compared by splitting the version strings
 * into components using the {@link #DELIMITER_PATTERN} and comparing each
 * component in order. The first difference found when comparing components
 * left-to-right is returned.
 * <p>
 * Two numeric components (containing only digits) are compared as integers. A
 * numeric component comes after any non-numeric one. Two non-numeric components
 * are ordered by {@link String#compareToIgnoreCase(String)}.
 */
public class VersionStringComparator implements Comparator<String> {
    public static final String DELIMITER_PATTERN = "[\\.-]";
    public static final String COMPONENT_PATTERN = "[\\d\\w]+";
    public static final String VALID_VERSION_PATTERN = COMPONENT_PATTERN + "(?:" + DELIMITER_PATTERN + COMPONENT_PATTERN + ")*";
    private static final Pattern START_WITH_INT_PATTERN = Pattern.compile("(^\\d+)");
    public static final Pattern SNAPSHOT_PATTERN = Pattern.compile(".*-SNAPSHOT$");

    public static boolean isValidVersionString(final String version) {
        return (version != null) && version.matches(VALID_VERSION_PATTERN);
    }

    public static boolean isSnapshotVersion(final String version) {
        return (version != null) && SNAPSHOT_PATTERN.matcher(version).matches();
    }

    /**
     * Compares two version strings. If either argument is not a String,
     * this method returns 0.
     *
     * @throws IllegalArgumentException if either argument is a String,
     * but does not match {@link #VALID_VERSION_PATTERN}.
     * @see #isValidVersionString(String)
     */
    //    public int compare(Object o1, Object o2)
    //    {
    //        if (!(o1 instanceof String)) return 0;
    //        if (!(o2 instanceof String)) return 0;
    //
    //        return compare((String) o1, (String) o2);
    //    }

    /**
     * Compares two version strings using the algorithm described above.
     *
     * @return <tt>-1</tt> if version1 is before version2, <tt>1</tt> if version2 is before
     * version1, or <tt>0</tt> if the versions are equal.
     * @throws IllegalArgumentException if either argument does not match {@link #VALID_VERSION_PATTERN}.
     * @see #isValidVersionString(String)
     */
    public int compare(final String version1, final String version2) {
        // Get the version numbers, remove all whitespaces
        String thisVersion = "0";
        if (StringUtils.isNotEmpty(version1)) {
            thisVersion = version1.replaceAll(" ", "");
        }
        String compareVersion = "0";
        if (StringUtils.isNotEmpty(version2)) {
            compareVersion = version2.replaceAll(" ", "");
        }

        if (!thisVersion.matches(VALID_VERSION_PATTERN) || !compareVersion.matches(VALID_VERSION_PATTERN)) {
            throw new IllegalArgumentException("Version number '" + thisVersion + "' cannot be compared to '" + compareVersion + "'");
        }

        // Split the version numbers
        final String[] v1 = thisVersion.split(DELIMITER_PATTERN);
        final String[] v2 = compareVersion.split(DELIMITER_PATTERN);

        final Comparator<String> componentComparator = new VersionStringComponentComparator();

        // Compare each place, until we find a difference and then return. If empty, assume zero.
        for (int i = 0; i < (v1.length > v2.length ? v1.length : v2.length); i++) {
            final String component1 = i >= v1.length ? "0" : v1[i];
            final String component2 = i >= v2.length ? "0" : v2[i];

            if (componentComparator.compare(component1, component2) != 0) {
                return componentComparator.compare(component1, component2);
            }
        }

        return 0;
    }

    private class VersionStringComponentComparator implements Comparator<String> {
        public static final int FIRST_GREATER = 1;
        public static final int SECOND_GREATER = -1;

        public int compare(final String component1, final String component2) {
            if (component1.equalsIgnoreCase(component2)) {
                return 0;
            }

            if (isInteger(component1) && isInteger(component2)) {
                return new BigInteger(component1).compareTo(new BigInteger(component2));
            }

            // Handles the case where we are comparing 1.5 to 1.6a
            final BigInteger comp1BigIntPart = getStartingInteger(component1);
            final BigInteger comp2BigIntPart = getStartingInteger(component2);
            if (comp1BigIntPart != null && comp2BigIntPart != null) {
                if (comp1BigIntPart.compareTo(comp2BigIntPart) > 0) {
                    return FIRST_GREATER;
                }

                if (comp2BigIntPart.compareTo(comp1BigIntPart) > 0) {
                    return SECOND_GREATER;
                }
            }

            // 2.3-alpha < 2.3.0 and 2.3a < 2.3
            // fixes PLUG-672. We are safe to do the integer check here since above we have
            // already determined that one of the two components are not an integer and that one does not start with
            // an int that may be larger than the other component
            if (isInteger(component1)) {
                return FIRST_GREATER;
            }
            if (isInteger(component2)) {
                return SECOND_GREATER;
            }

            // 2.3a < 2.3b
            return component1.compareToIgnoreCase(component2);
        }

        private boolean isInteger(final String string) {
            return string.matches("\\d+");
        }

        private BigInteger getStartingInteger(final String string) {
            Matcher matcher = START_WITH_INT_PATTERN.matcher(string);
            if (matcher.find()) {
                // If we found a starting digit group then lets return it
                return new BigInteger(matcher.group(1));
            }
            return null;
        }

    }
}
