package com.atlassian.plugin;

import java.io.InputStream;
import java.io.File;

/**
 * Allows the retrieval of files and/or an input stream of a plugin artifact. Implementations
 * must allow multiple calls to {@link #getInputStream()}.
 *
 * @see PluginController
 * @since 2.0.0
 */
public interface PluginArtifact
{
    /**
     * @return true if the resource exists in this artifact, otherwise false
     * @since 2.2.0
     */
    boolean doesResourceExist(String name);

    /**
     * @return an input stream of the resource specified inside the artifact.  Null if the resource cannot be found.
     * @throws PluginParseException if the there was an exception retrieving the resource from the artifact
     */
    InputStream getResourceAsStream(String name) throws PluginParseException;

    /**
     * @return the original name of the plugin artifact file. Typically used
     * for persisting it to disk with a meaningful name.
     */
    String getName();

    /**
     * @return an InputStream for the entire plugin artifact. Calling this
     * multiple times will return a fresh input stream each time.
     */
    InputStream getInputStream();

    /**
     * @return the artifact as a file, or its underlying file if it is already one
     * @since 2.2.0
     */
    File toFile();

    /**
     * @return {@code true} if the plugin contains or references java executable code.
     * @since 3.0
     */
    boolean containsJavaExecutableCode();

    /**
     * Additional interface for a plugin artifact which may support reference installation.
     */
    interface AllowsReference
    {
        /**
         * A named boolean for specifying {@link PluginArtifact.AllowsReference#allowsReference}
         * return values.
         */
        enum ReferenceMode
        {
            /** {@link AllowsReference#allowsReference} returns false. */
            FORBID_REFERENCE(false),

            /** {@link PluginArtifact.AllowsReference#allowsReference} returns true. */
            PERMIT_REFERENCE(true);

            private boolean allowsReference;

            private ReferenceMode(boolean allowsReference)
            {
                this.allowsReference = allowsReference;
            }

            boolean allowsReference()
            {
                return allowsReference;
            }
        }

        /**
         * Query whether or not the PluginArtifact may be reference installed.
         *
         * @return true iff the artifact may be reference installed.
         */
        boolean allowsReference();

        /**
         * Host class for a static accessor which defaults allowsReference to false.
         */
        class Default
        {
            /**
             * Determine if a PluginArtifact allows reference installation.
             *
             * If the PluginArtifact supports AllowsReference, it's allowsReference is queried, otherwise
             * it defaults to false.
             *
             * @param pluginArtifact the plugin artifact to check.
             * @return true iff a reference install should be attempted for pluginArtifact.
             */
            public static boolean allowsReference(PluginArtifact pluginArtifact)
            {
                return (pluginArtifact instanceof PluginArtifact.AllowsReference) &&
                        ((PluginArtifact.AllowsReference) pluginArtifact).allowsReference();
            }
        }
    }
}
