package com.atlassian.plugin.parsers;

import com.atlassian.fugue.Option;
import com.atlassian.plugin.Application;
import com.atlassian.plugin.InstallationMode;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.ModuleDescriptorFactory;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginInformation;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.PluginPermission;
import com.atlassian.plugin.Resources;
import com.atlassian.plugin.descriptors.UnloadableModuleDescriptor;
import com.atlassian.plugin.descriptors.UnrecognisedModuleDescriptor;
import com.atlassian.plugin.impl.UnloadablePluginFactory;
import com.atlassian.plugin.util.PluginUtils;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import static com.atlassian.plugin.descriptors.UnloadableModuleDescriptorFactory.createUnloadableModuleDescriptor;
import static com.atlassian.plugin.descriptors.UnrecognisedModuleDescriptorFactory.createUnrecognisedModuleDescriptor;
import static com.atlassian.plugin.parsers.XmlDescriptorParserUtils.removeAllNamespaces;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;

/**
 * Provides access to the descriptor information retrieved from an XML InputStream.
 * <p/>
 * Uses the dom4j {@link SAXReader} to parse the XML stream into a document
 * when the parser is constructed.
 *
 * @see XmlDescriptorParserFactory
 */
public class XmlDescriptorParser implements DescriptorParser
{
    private static final Logger log = LoggerFactory.getLogger(XmlDescriptorParser.class);

    private final Document document;
    private final Set<Application> applications;

    /**
     * Constructs a parser with an already-constructed document
     * @param source the source document
     * @param applications the application key to filter modules with, null for all unspecified
     * @throws PluginParseException if there is a problem reading the descriptor from the XML {@link InputStream}.
     * @since 2.2.0
     */
    public XmlDescriptorParser(final Document source, final Set<Application> applications) throws PluginParseException
    {
        this.document = removeAllNamespaces(checkNotNull(source, "XML descriptor source document cannot be null"));
        this.applications = ImmutableSet.copyOf(checkNotNull(applications));
    }

    /**
     * Constructs a parser with a stream of an XML document for a specific application
     * @param source The descriptor stream
     * @param applications the application key to filter modules with, null for all unspecified
     * @throws PluginParseException if there is a problem reading the descriptor from the XML {@link InputStream}.
     * @since 2.2.0
     */
    public XmlDescriptorParser(final InputStream source, final Set<Application> applications) throws PluginParseException
    {
        this(createDocument(checkNotNull(source, "XML descriptor source cannot be null")), applications);
    }

    protected static Document createDocument(final InputStream source) throws PluginParseException
    {
        final SAXReader reader = new SAXReader();
        reader.setMergeAdjacentText(true);
        try
        {
            return reader.read(source);
        }
        catch (final DocumentException e)
        {
            throw new PluginParseException("Cannot parse XML plugin descriptor", e);
        }
    }

    protected Document getDocument()
    {
        return document;
    }

    public Plugin configurePlugin(final ModuleDescriptorFactory moduleDescriptorFactory, final Plugin plugin) throws PluginParseException
    {
        final Element pluginElement = getPluginElement();
        plugin.setName(pluginElement.attributeValue("name"));
        plugin.setKey(getKey());
        plugin.setPluginsVersion(getPluginsVersion());
        plugin.setSystemPlugin(isSystemPlugin());

        if (pluginElement.attributeValue("i18n-name-key") != null)
        {
            plugin.setI18nNameKey(pluginElement.attributeValue("i18n-name-key"));
        }

        if (plugin.getKey().indexOf(":") > 0)
        {
            throw new PluginParseException("Plugin keys cannot contain ':'. Key is '" + plugin.getKey() + "'");
        }

        if ("disabled".equalsIgnoreCase(pluginElement.attributeValue("state")))
        {
            plugin.setEnabledByDefault(false);
        }

        plugin.setResources(Resources.fromXml(pluginElement));

        for (final Iterator i = pluginElement.elementIterator(); i.hasNext(); )
        {
            final Element element = (Element) i.next();

            if ("plugin-info".equalsIgnoreCase(element.getName()))
            {
                plugin.setPluginInformation(createPluginInformation(element));
            }
            else if (!"resource".equalsIgnoreCase(element.getName()))
            {
                final ModuleDescriptor<?> moduleDescriptor = createModuleDescriptor(plugin, element, moduleDescriptorFactory);

                // If we're not loading the module descriptor, null is returned, so we skip it
                if (moduleDescriptor == null)
                {
                    continue;
                }

                if (plugin.getModuleDescriptor(moduleDescriptor.getKey()) != null)
                {
                    throw new PluginParseException("Found duplicate key '" + moduleDescriptor.getKey() + "' within plugin '" + plugin.getKey() + "'");
                }

                plugin.addModuleDescriptor(moduleDescriptor);

                // If we have any unloadable modules, also create an unloadable plugin, which will make it clear that there was a problem
                if (moduleDescriptor instanceof UnloadableModuleDescriptor)
                {
                    log.error("There were errors loading the plugin '" + plugin.getName() + "'. The plugin has been disabled.");
                    return UnloadablePluginFactory.createUnloadablePlugin(plugin);
                }
            }
        }

        return plugin;
    }

    private Element getPluginElement()
    {
        return document.getRootElement();
    }

    protected ModuleDescriptor<?> createModuleDescriptor(final Plugin plugin, final Element element, final ModuleDescriptorFactory moduleDescriptorFactory) throws PluginParseException
    {
        final String name = element.getName();

        // Determine if this module descriptor is applicable for the current application
        if (!PluginUtils.doesModuleElementApplyToApplication(element, applications, plugin.getInstallationMode()))
        {
            log.debug("Ignoring module descriptor for this application: " + element.attributeValue("key"));
            return null;
        }

        ModuleDescriptor<?> moduleDescriptorDescriptor;

        // Try to retrieve the module descriptor
        try
        {
            moduleDescriptorDescriptor = moduleDescriptorFactory.getModuleDescriptor(name);
        }
        // When there's a problem loading a module, return an UnrecognisedModuleDescriptor with error
        catch (final Throwable e)
        {
            final UnrecognisedModuleDescriptor descriptor = createUnrecognisedModuleDescriptor(plugin, element, e, moduleDescriptorFactory);

            log.error("There were problems loading the module '" + name + "' in plugin '" + plugin.getName() + "'. The module has been disabled.");
            log.error(descriptor.getErrorText(), e);

            return descriptor;
        }

        // When the module descriptor has been excluded, null is returned (PLUG-5)
        if (moduleDescriptorDescriptor == null)
        {
            log.info("The module '" + name + "' in plugin '" + plugin.getName() + "' is in the list of excluded module descriptors, so not enabling.");
            return null;
        }

        // Once we have the module descriptor, create it using the given information
        try
        {
            moduleDescriptorDescriptor.init(plugin, element);
        }
        // If it fails, return a dummy module that contains the error
        catch (final Exception e)
        {
            final UnloadableModuleDescriptor descriptor = createUnloadableModuleDescriptor(plugin, element, e, moduleDescriptorFactory);

            log.error("There were problems loading the module '" + name + "'. The module and its plugin have been disabled.");
            log.error(descriptor.getErrorText(), e);

            return descriptor;
        }

        return moduleDescriptorDescriptor;
    }

    protected PluginInformation createPluginInformation(final Element element)
    {
        final PluginInformation pluginInfo = new PluginInformation();

        if (element.element("description") != null)
        {
            pluginInfo.setDescription(element.element("description").getTextTrim());
            if (element.element("description").attributeValue("key") != null)
            {
                pluginInfo.setDescriptionKey(element.element("description").attributeValue("key"));
            }
        }

        if (element.element("version") != null)
        {
            pluginInfo.setVersion(element.element("version").getTextTrim());
        }

        if (element.element("vendor") != null)
        {
            final Element vendor = element.element("vendor");
            pluginInfo.setVendorName(vendor.attributeValue("name"));
            pluginInfo.setVendorUrl(vendor.attributeValue("url"));
        }

        // initialize any parameters on the plugin xml definition
        for (Element param : getElements(element, "param"))
        {
            // Retrieve the parameter info => name & text
            if (param.attribute("name") != null)
            {
                pluginInfo.addParameter(param.attribute("name").getData().toString(), param.getText());
            }
        }

        if (element.element("application-version") != null)
        {
            final Element ver = element.element("application-version");
            if (ver.attribute("max") != null)
            {
                pluginInfo.setMaxVersion(Float.parseFloat(ver.attributeValue("max")));
            }
            if (ver.attribute("min") != null)
            {
                pluginInfo.setMinVersion(Float.parseFloat(ver.attributeValue("min")));
            }
        }

        if (element.element("java-version") != null)
        {
            pluginInfo.setMinJavaVersion(Float.valueOf(element.element("java-version").attributeValue("min")));
        }

        if (element.element("permissions") != null)
        {
            ImmutableSet.Builder<PluginPermission> permissions = ImmutableSet.builder();
            for (Element permission : filter(getElements(element.element("permissions"), "permission"), new ElementWithForApplicationsPredicate(applications)))
            {
                final String trimmedPermission = permission.getTextTrim();
                if (StringUtils.isNotBlank(trimmedPermission))
                {
                    final String parsedInstallationMode = permission.attributeValue("installation-mode");
                    final Option<InstallationMode> installationMode = InstallationMode.of(parsedInstallationMode);
                    if (StringUtils.isNotBlank(parsedInstallationMode) && !installationMode.isDefined())
                    {
                        log.warn("The parsed installation mode '{}' for permission '{}' didn't match any of the valid values: {}",
                                new Object[]{parsedInstallationMode, trimmedPermission,
                                        transform(copyOf(InstallationMode.values()), new Function<InstallationMode, String>()
                                        {
                                            @Override
                                            public String apply(InstallationMode im)
                                            {
                                                return im.getKey();
                                            }
                                        })});
                    }

                    permissions.add(new PluginPermission(trimmedPermission, installationMode));
                }
                else
                {
                    log.warn("Plugin {} has blank permission.", getKey());
                }
            }
            pluginInfo.setPermissions(permissions.build());
        }
        else if (getPluginsVersion() < Plugin.VERSION_3)
        {
            pluginInfo.setPermissions(ImmutableSet.of(PluginPermission.ALL));
        }

        return pluginInfo;
    }

    public String getKey()
    {
        return getPluginElement().attributeValue("key");
    }

    public int getPluginsVersion()
    {
        String val = getPluginElement().attributeValue("pluginsVersion");
        if (val == null)
        {
            val = getPluginElement().attributeValue("plugins-version");
        }
        if (val != null)
        {
            try
            {
                return Integer.parseInt(val);
            }
            catch (final NumberFormatException e)
            {
                throw new RuntimeException("Could not parse pluginsVersion: " + e.getMessage(), e);
            }
        }
        else
        {
            return 1;
        }
    }

    public PluginInformation getPluginInformation()
    {
        return createPluginInformation(getDocument().getRootElement().element("plugin-info"));
    }

    public boolean isSystemPlugin()
    {
        return Boolean.valueOf(getPluginElement().attributeValue("system"));
    }

    @SuppressWarnings("unchecked")
    private List<Element> getElements(Element element, String name)
    {
        return element.elements(name);
    }

    private static final class ApplicationWithNamePredicate implements Predicate<Application>
    {
        private final String name;

        public ApplicationWithNamePredicate(String name)
        {
            this.name = name;
        }

        @Override
        public boolean apply(Application app)
        {
            return app.getKey().equals(name); // name might be null
        }
    }

    private static final class ElementWithForApplicationsPredicate implements Predicate<Element>
    {
        private final Set<Application> applications;

        private ElementWithForApplicationsPredicate(Set<Application> applications)
        {
            this.applications = checkNotNull(applications);
        }

        @Override
        public boolean apply(final Element el)
        {
            final String appName = el.attributeValue("application");
            return appName == null || Iterables.any(applications, new ApplicationWithNamePredicate(appName));
        }
    }
}
