package com.atlassian.plugin.loaders;

import com.atlassian.plugin.ModuleDescriptorFactory;
import com.atlassian.plugin.Permissions;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginArtifact;
import com.atlassian.plugin.PluginArtifactBackedPlugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.impl.UnloadablePlugin;
import com.google.common.base.Function;

import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Iterables.transform;

public final class PermissionCheckingPluginLoader extends ForwardingPluginLoader
{
    public PermissionCheckingPluginLoader(PluginLoader delegate)
    {
        super(delegate);
    }

    @Override
    public Iterable<Plugin> loadAllPlugins(ModuleDescriptorFactory moduleDescriptorFactory) throws PluginParseException
    {
        return copyOf(transform(delegate().loadAllPlugins(moduleDescriptorFactory), new CheckPluginPermissionFunction()));
    }

    @Override
    public Iterable<Plugin> loadFoundPlugins(ModuleDescriptorFactory moduleDescriptorFactory) throws PluginParseException
    {
        return copyOf(transform(delegate().loadFoundPlugins(moduleDescriptorFactory), new CheckPluginPermissionFunction()));
    }

    private static final class CheckPluginPermissionFunction implements Function<Plugin, Plugin>
    {
        @Override
        public Plugin apply(Plugin p)
        {
            // we don't care if we don't need execute java
            if (!p.getActivePermissions().contains(Permissions.EXECUTE_JAVA))
            {
                return p;
            }

            if (p instanceof PluginArtifactBackedPlugin)
            {
                return checkPlugin((PluginArtifactBackedPlugin) p);
            }
            return p;
        }

        private Plugin checkPlugin(PluginArtifactBackedPlugin p)
        {
            final PluginArtifact pluginArtifact = p.getPluginArtifact();

            if (pluginArtifact.containsJavaExecutableCode())
            {
                UnloadablePlugin unloadablePlugin = new UnloadablePlugin(
                        "Plugin doesn't require '" + Permissions.EXECUTE_JAVA + "' permission yet references some java executable code. " +
                        "This could be either embedded java classes, embedded java libraries, spring context files or bundle activator.");
                unloadablePlugin.setKey(p.getKey());
                unloadablePlugin.setName(p.getName());
                return unloadablePlugin;
            }
            else
            {
                return p;
            }
        }
    }
}
