package com.atlassian.plugin.factories;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.ModuleDescriptorFactory;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginArtifact;
import com.atlassian.plugin.module.Element;

/**
 * Creates the plugin artifact and deploys it into the appropriate plugin management system
 *
 * @since 2.0.0
 */
public interface PluginFactory {
    /**
     * Determines if this factory can handle this artifact.
     *
     * @param pluginArtifact The artifact to test
     * @return The plugin key, null if it cannot load the plugin
     * @throws com.atlassian.plugin.PluginParseException If there are exceptions parsing the plugin configuration when
     *                                                   the deployer should have been able to deploy the plugin
     */
    String canCreate(PluginArtifact pluginArtifact);

    /**
     * Deploys the plugin artifact by instantiating the plugin and configuring it. Should only be called if the respective
     * {@link #canCreate(PluginArtifact)} call returned the plugin key
     *
     * @param pluginArtifact          the plugin artifact to deploy
     * @param moduleDescriptorFactory the factory for the module descriptors
     * @return the plugin loaded from the plugin artifact, or an UnloadablePlugin instance if loading fails.
     * @throws com.atlassian.plugin.PluginParseException if the plugin could not be parsed
     * @since 2.2.0
     */
    Plugin create(PluginArtifact pluginArtifact, ModuleDescriptorFactory moduleDescriptorFactory);

    /**
     * If this factory is capable of loading a plugin of the type passed, attempt to create a module descriptor. Add that
     * module descriptor to the plugin.
     * <p>
     * If capable, always return a ModuleDescriptor, even if it indicates a failure case. Caller is responsible for
     * handling exceptional {@link com.atlassian.plugin.ModuleDescriptor}.
     * <p>
     * Implementors should use <code>(plugin instanceof &lt;plugin-class&gt;)</code> to determine their capability;
     * subclasses of implementors should override this if special handling is required.
     *
     * @param plugin                  that the module will be a member of
     * @param module                  to create
     * @param moduleDescriptorFactory basic factory, may be overridden
     * @return null plugin was not created by this PluginFactory
     * @since 4.0.0
     */
    ModuleDescriptor<?> createModule(Plugin plugin, Element module, ModuleDescriptorFactory moduleDescriptorFactory);
}
