define('atlassian-nps-plugin/js/nps/server/btf-admin', [
    'jquery',
    'atlassian-nps-plugin/js/amd-shims/ajs',
    'atlassian-nps-plugin/js/amd-shims/templates',
    'atlassian-nps-plugin/js/nps/server/nps-status-manager',
    'atlassian-nps-plugin/js/nps/survey-dialog',
    'atlassian-nps-plugin/js/nps/product',
    'atlassian-nps-plugin/js/nps/submission'
],
/**
 * On Server instances, system administrators can turn NPS on/off via the analytics admin page
 * where there is a NPS checkbox indicating the NPS state on the instance. They can also preview
 * what the survey dialog will look like when presented to users.
 * @param $ jQuery
 * @param AJS
 * @param Templates NPS admin and survey preview templates
 * @param NpsStatusManager manager for handling turning NPS on/off
 * @param SurveyDialog sample model dialog with the NPS survey
 * @param SurveyInlineDialog sample inline dialog with the NPS survey, SurveyDialog will be replaced by this inline dialog if nps.survey.inline.dialog dark feature is enabled
 * @param Product adapter for a specific product such as JIRA or Confluence
 * @param Submission handles the firing of analytics events
 */
function(
    $,
    AJS,
    Templates,
    NpsStatusManager,
    SurveyDialog,
    Product,
    Submission
) {
    'use strict';

    function init($form) {
        NpsStatusManager.getNpsStatus().then(function(isNpsEnabled) {
            var oldNpsState = isNpsEnabled;

            var $radioButtons = $form.find('input.radio');
            var $enableAnalyticsRadioButton = $form.find('#enable-analytics');
            var isAnalyticsEnabled = $enableAnalyticsRadioButton.is(':checked');

            displayNpsSection(isNpsEnabled, $form, isAnalyticsEnabled);
            var $enableNpsCheckbox = $form.find('#enable-nps');

            // update whether or NPS can be modified depending on if analytics is enabled
            $radioButtons.change(function() {
                $enableNpsCheckbox.prop('disabled', !$enableAnalyticsRadioButton.is(':checked'));
                $enableNpsCheckbox.prop('checked', false);
            });

            var $previewLink = $('#nps-preview');
            if (AJS.DarkFeatures && AJS.DarkFeatures.isEnabled("nps.survey.inline.dialog")) {
                // It's difficult to declare Dark Feature cross products and try to avoid loading redundant dependency (inline dialog 2)
                require(['atlassian-nps-plugin/js/nps/survey-inline-dialog'], function(SurveyInlineDialog) {
                    $previewLink.on('click', SurveyInlineDialog.preview);
                });
            } else {
                $previewLink.on('click', SurveyDialog.previewDialog);
            }

            // wait for analytics to successfully update before making any NPS changes
            $(document).ajaxSuccess(function(event, ajaxOptions, data) {
                if (data.url === AJS.contextPath() + '/rest/analytics/1.0/config/enable') {
                    var newNpsState = $form.find('input[name=enable-nps]:checked').val() === 'on';
                    // Only try to update NPS if its config has changed
                    if (oldNpsState !== newNpsState) {
                        NpsStatusManager.updateNpsStatus(newNpsState, {useErrorFlags: true}).then(function() {
                            Submission.updateNpsStatus({ enabled: newNpsState });
                            oldNpsState = newNpsState;
                        });
                    }
                }
            });
        });
    }

    function displayNpsSection(isEnabled, $form, isAnalyticsEnabled) {
        var btfAdminContent = Templates.btfAdmin({
            status: isEnabled,
            isDisabled: !isAnalyticsEnabled,
            previewLink: Templates.previewLink(),
            productName: Product.getProductName()
        });
        var $atlassianAnalytics = $form.find('fieldset');
        $atlassianAnalytics.after(btfAdminContent);
    }

    return {
        init: init
    };
});
