package com.atlassian.plugins.navlink.producer.navigation.services;

import com.atlassian.applinks.host.spi.InternalHostApplication;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLinks;
import com.atlassian.plugins.navlink.util.i18n.LocaleSupportingI18nResolverWorkAround;
import com.atlassian.plugins.navlink.util.url.UrlFactory;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Locale;
import java.util.Set;

public class LocalNavigationLinkServiceImpl implements LocalNavigationLinkService
{
    private final LocalNavigationLinks localNavigationLinks;
    private final UrlFactory urlFactory;
    private final InternalHostApplication internalHostApplication;
    private final LocaleSupportingI18nResolverWorkAround i18nResolverWorkAround;

    public LocalNavigationLinkServiceImpl(LocalNavigationLinks localNavigationLinks,
            UrlFactory urlFactory,
            InternalHostApplication internalHostApplication,
            LocaleSupportingI18nResolverWorkAround i18nResolverWorkAround)
    {
        this.localNavigationLinks = localNavigationLinks;
        this.urlFactory = urlFactory;
        this.internalHostApplication = internalHostApplication;
        this.i18nResolverWorkAround = i18nResolverWorkAround;
    }
    @Override
    @Nonnull
    public Set<NavigationLink> all(@Nonnull final Locale locale)
    {
        final Set<RawNavigationLink> all = localNavigationLinks.all();
        return Sets.newHashSet(Iterables.transform(all, toNavigationLinkEntity(locale)));
    }

    @Override
    @Nonnull
    public Set<NavigationLink> matching(@Nonnull final Locale locale, @Nonnull Predicate<NavigationLink> criteria)
    {
        return Sets.newHashSet(Iterables.filter(all(locale), criteria));
    }

    @Nonnull
    private Function<? super RawNavigationLink, ? extends NavigationLink> toNavigationLinkEntity(@Nonnull final Locale locale)
    {
        return new Function<RawNavigationLink, NavigationLink>()
        {
            @Override
            public NavigationLink apply(@Nullable RawNavigationLink localLink)
            {
                if (localLink != null)
                {
                    final String href = urlFactory.toAbsoluteUrl(localLink.getHref());
                    final String label = i18nResolverWorkAround.getText(locale, localLink.getLabelKey(), internalHostApplication.getName());
                    final String tooltip = i18nResolverWorkAround.getText(locale, localLink.getTooltipKey());
                    final String icon = Strings.isNullOrEmpty(localLink.getIconUrl()) ? null : urlFactory.toAbsoluteUrl(localLink.getIconUrl());
                    return NavigationLinks.copyOf(localLink)
                            .href(href)
                            .iconUrl(icon)
                            .label(label)
                            .tooltip(tooltip)
                            .build();
                }
                else
                {
                    return null;
                }
            }
        };
    }

}
