package com.atlassian.plugins.navlink.producer.capabilities.rest;

import com.atlassian.plugins.navlink.producer.capabilities.ApplicationWithCapabilities;
import com.atlassian.plugins.navlink.producer.capabilities.services.CapabilityService;
import com.atlassian.plugins.navlink.util.JsonStringEncoder;
import com.atlassian.plugins.navlink.util.LastModifiedFormatter;
import com.atlassian.plugins.navlink.util.date.UniversalDateFormatter;
import com.atlassian.plugins.navlink.util.url.BaseUrl;
import com.atlassian.plugins.navlink.util.url.SelfUrl;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.templaterenderer.TemplateRenderer;
import com.google.common.collect.ImmutableMap;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.HttpHeaders;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.Date;
import java.util.Map;

import static com.atlassian.plugins.navlink.util.CacheControlFactory.withConfiguredMaxAgeAndStaleContentExtension;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static org.apache.commons.lang.CharEncoding.UTF_8;

/**
 * Provides REST data about Application Capabilities.
 *
 * @since 2.0.3
 */
public class CapabilitiesServlet extends HttpServlet
{
    private static final String VELOCITY_TEMPLATES = "templates/capabilities.vm";

    private final Logger logger = LoggerFactory.getLogger(CapabilitiesServlet.class);
    private final CapabilityService capabilityService;
    private final TemplateRenderer templateRenderer;
    private final ApplicationProperties applicationProperties;

    public CapabilitiesServlet(final CapabilityService capabilityService, final TemplateRenderer templateRenderer,
                               ApplicationProperties applicationProperties)
    {
        this.capabilityService = capabilityService;
        this.templateRenderer = templateRenderer;
        this.applicationProperties = applicationProperties;
    }

    @Override
    protected void doGet(final HttpServletRequest httpServletRequest, final HttpServletResponse httpServletResponse) throws ServletException, IOException
    {
        try
        {
            httpServletResponse.setContentType(APPLICATION_JSON);
            httpServletResponse.setCharacterEncoding(UTF_8);
            httpServletResponse.setHeader(HttpHeaders.CACHE_CONTROL, withConfiguredMaxAgeAndStaleContentExtension().toString());
            httpServletResponse.setHeader(HttpHeaders.LAST_MODIFIED, LastModifiedFormatter.formatCurrentTimeMillis());

            final Map<String, Object> context = createContext(httpServletRequest);
            final PrintWriter writer = httpServletResponse.getWriter();
            renderTemplate(context, writer);
        }
        catch (IOException e)
        {
            handleException(httpServletResponse, e);
        }
    }

    private void renderTemplate(final Map<String, Object> context, final Writer writer) throws IOException
    {
        templateRenderer.render(VELOCITY_TEMPLATES, context, writer);
    }

    private Map<String, Object> createContext(final HttpServletRequest httpServletRequest)
    {
        final BaseUrl baseUrl = BaseUrl.extractFrom(httpServletRequest);
        final String selfUrl = SelfUrl.extractFrom(httpServletRequest);
        final ApplicationWithCapabilities hostApplication = capabilityService.getHostApplication();
        return ImmutableMap.<String, Object>builder()
                .put("baseUrl", baseUrl)
                .put("selfUrl", selfUrl)
                .put("buildDate", formatBuildDate())
                .put("hostApplication", hostApplication)
                .put("json", new JsonStringEncoder())
                .build();
    }

    private void handleException(final HttpServletResponse httpServletResponse, final Exception e)
    {
        logger.warn("Failed to serialize application capabilities: {}", e.getMessage());
        logger.debug("Stacktrace:", e);
        httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
    }

    private String formatBuildDate()
    {
        final Date buildDate = applicationProperties.getBuildDate();
        return UniversalDateFormatter.formatUtc(new DateTime(buildDate));
    }

}
