package com.atlassian.plugins.navlink.consumer.projectshortcuts.rest;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;

import com.atlassian.applinks.spi.application.TypeId;
import com.atlassian.fugue.Pair;
import com.atlassian.plugins.navlink.producer.contentlinks.customcontentlink.CustomContentLinkService;
import com.atlassian.plugins.navlink.producer.contentlinks.plugin.ContentLinkModuleDescriptor;
import com.atlassian.plugins.navlink.producer.contentlinks.rest.ContentLinkEntity;
import com.atlassian.plugins.navlink.producer.contentlinks.rest.ContentLinkEntityFactory;
import com.atlassian.plugins.navlink.producer.contentlinks.services.ContentLinksService;
import com.atlassian.plugins.navlink.util.url.UrlFactory;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.lang.StringUtils;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

@Path("/")
public class ProjectShortcutsResource
{
    private ContentLinksService contentLinksService;
    private CustomContentLinkService customContentLinksService;

    private UrlFactory urlFactory;

    public ProjectShortcutsResource(ContentLinksService contentLinksService,
            CustomContentLinkService customContentLinksService, UrlFactory urlFactory)
    {
        this.contentLinksService = contentLinksService;
        this.customContentLinksService = customContentLinksService;
        this.urlFactory = urlFactory;
    }

    /**
     * Returns the project information and local links for the project. Will not return any information if user can not
     * see the project.
     */
    @GET
    @Path("local/{key}")
    @Produces(APPLICATION_JSON)
    @AnonymousAllowed
    public Response getLocalContentLinksAndProjectDetails(@PathParam("key") String key,
            @Nullable @QueryParam("entityType") String entityType, @Context HttpServletRequest request)
    {
        if (StringUtils.isBlank(key))
        {
            return Response.status(Response.Status.BAD_REQUEST).entity("Please provide a project key").build();
        }

        Map<String, Object> context = Maps.newHashMap();
        context.put("key", key);
        // todo do we want all web params added?

        List<ContentLinkModuleDescriptor> localContentLinksDescriptors = contentLinksService.getAllLocalContentLinks(
                context, entityType != null ? new TypeId(entityType) : null);
        ContentLinkEntityFactory factory = new ContentLinkEntityFactory(request, context, urlFactory);

        List<ContentLinkEntity> contentLinkEntities = new ArrayList<ContentLinkEntity>(factory.create(localContentLinksDescriptors));

        contentLinkEntities.addAll(factory.createFromCustomContentLinks(customContentLinksService.getPluginCustomContentLinks(key), false));

        return Response.ok(new ProjectShortCutEnvelope(contentLinkEntities, Collections.<UnauthenticatedRemoteApplication>emptyList())).build();
    }

    /**
     * Return content links from any connected project/app
     */
    @GET
    @Path("remote/{key}")
    @Produces(APPLICATION_JSON)
    @AnonymousAllowed
    public Response getRemoteContentLinks(@PathParam("key") String key, @QueryParam("entityType") String entityType,
            @Context HttpServletRequest request)
    {
        if (StringUtils.isBlank(key))
        {
            return Response.status(Response.Status.BAD_REQUEST).entity("Please provide a project key").build();
        }

        if (StringUtils.isBlank(entityType))
        {
            return Response.status(Response.Status.BAD_REQUEST).entity("Please provide an entityType").build();
        }

        Map<String, Object> context = Maps.newHashMap();
        context.put("key", key);

        final Pair<Iterable<ContentLinkEntity>,Iterable<UnauthenticatedRemoteApplication>> allRemoteContentLinksAndUnauthedApps = contentLinksService.getAllRemoteContentLinksAndUnauthedApps(key, new TypeId(entityType));

        List<ContentLinkEntity> remoteContentLinks = Lists.newArrayList(allRemoteContentLinksAndUnauthedApps.left());

        remoteContentLinks.addAll(new ContentLinkEntityFactory(request, context, urlFactory)
            .createFromCustomContentLinks(customContentLinksService.getCustomContentLinks(key), false));
        
        return Response.ok(new ProjectShortCutEnvelope(remoteContentLinks, Lists.newArrayList(allRemoteContentLinksAndUnauthedApps.right()))).build();
    }
}
