package com.atlassian.plugins.navlink.consumer.menu.services;

import com.atlassian.plugins.custom_apps.api.CustomApp;
import com.atlassian.plugins.custom_apps.api.CustomAppService;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLinkBuilder;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLinkPredicates;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLinks;
import com.atlassian.plugins.navlink.producer.navigation.services.LocalNavigationLinkService;
import com.atlassian.sal.api.user.UserManager;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import static com.atlassian.plugins.navlink.producer.navigation.NavigationLinkPredicates.keyEquals;
import static com.google.common.base.Predicates.not;
import static com.google.common.collect.Iterables.transform;

public class DefaultMenuService implements MenuService
{
    private final LocalNavigationLinkService localNavigationLinkService;
    private final RemoteNavigationLinkService remoteNavigationLinkService;
    private final CustomAppService customAppService;
    private final UserManager userManager;

    public DefaultMenuService(final LocalNavigationLinkService localNavigationLinkService, final RemoteNavigationLinkService remoteNavigationLinkService, final CustomAppService customAppService, final UserManager userManager)
    {
        this.localNavigationLinkService = localNavigationLinkService;
        this.remoteNavigationLinkService = remoteNavigationLinkService;
        this.customAppService = customAppService;
        this.userManager = userManager;
    }

    @Nonnull
    @Override
    public Iterable<NavigationLink> getMenuItems(@Nonnull final String key, final String userName, @Nonnull final Locale locale)
    {
        final Set<NavigationLink> localNavigationLinks = getLocalNavigationLinks(key, locale);
        final Set<NavigationLink> remoteNavigationLinks = getRemoteNavigationLinks(key, locale);
        return mergeNavigationLinks(localNavigationLinks, remoteNavigationLinks);
    }

    @Override
    @Nonnull
    public Iterable<NavigationLink> getAppSwitcherItems(String userName)
    {
        List<CustomApp> apps = getVisibleLocalCustomAppsAndRemoteLinks(userName);
        List<NavigationLink> menu = Lists.newArrayListWithCapacity(apps.size());
        // convert to MenuNavigationLinks
        for (CustomApp customApp : apps)
        {
            menu.add(NavigationLinks.copyOf(customApp).key("home").build());
        }
        return menu;
    }

    /**
     * Return all local and remote navigation links and custom apps filtered by the hide flag and the group membership
     * of the user.
     * @param userName name of the user to check visibility against
     * @return a list of CustomApps including remote links for which we have defined an order and any new remote links,
     *  placed at the end, filtered to exclude those not visible to the user
     */
    @Nonnull
    private List<CustomApp> getVisibleLocalCustomAppsAndRemoteLinks(@Nullable final String userName)
    {
        final List<CustomApp> localAndRemoteLinks = customAppService.getLocalCustomAppsAndRemoteLinks();
        final Iterable<CustomApp> localAndRemoteLinksVisibleToUser = Iterables.filter(localAndRemoteLinks, isVisibleForUser(userName));
        return Lists.newArrayList(localAndRemoteLinksVisibleToUser);
    }

    @Nonnull
    private Predicate<CustomApp> isVisibleForUser(final String userName)
    {
        return new Predicate<CustomApp>()
        {
            @Override
            public boolean apply(@Nullable CustomApp input)
            {
                return input != null && !input.getHide() && visibleToCurrentUser(input, userName);
            }
        };
    }

    public boolean visibleToCurrentUser(final CustomApp customApp, @Nullable final String userName)
    {

        if (customApp.getAllowedGroups().isEmpty())
        {
            return true;
        }

        if(StringUtils.isBlank(userName))
        {
            // a null/anonymous user cannot see something with restricted group access.
            return false;
        }

        for (String groupName : customApp.getAllowedGroups())
        {
            if (userManager.isUserInGroup(userName, groupName))
            {
                return true;
            }
        }

        return false;
    }

    /**
     * Return <code>true</code> if the application navigator should be made visible to the given user. This is the case
     * if there is at least one link excluding the self link visible to the user.
     * @param userName name of the user to check visibility against
     * @return <code>true</code> if the application navigator should be made visible to the given user,
     * <code>false</code> otherwise
     */
    @Override
    public boolean isAppSwitcherVisibleForUser(@Nullable final String userName)
    {
        final List<CustomApp> localAndRemoteLinks = customAppService.getLocalCustomAppsAndRemoteLinks();
        return Iterables.any(localAndRemoteLinks, Predicates.and(not(isSelfLink()), isVisibleForUser(userName)));
    }

    @Nonnull
    private Predicate<CustomApp> isSelfLink()
    {
        return new Predicate<CustomApp>()
        {
            @Override
            public boolean apply(@Nullable final CustomApp customApp)
            {
                return customApp != null && customApp.isSelf();
            }
        };
    }

    @Nonnull
    private Set<NavigationLink> getLocalNavigationLinks(@Nonnull final String key, @Nonnull final Locale locale)
    {
        return localNavigationLinkService.matching(locale, NavigationLinkPredicates.keyEquals(key));
    }


    @Nonnull
    private Set<NavigationLink> getRemoteNavigationLinks(@Nonnull final String key, @Nonnull final Locale locale)
    {
        Set<NavigationLink> matches = remoteNavigationLinkService.matching(locale, keyEquals(key));
        return Sets.newHashSet(transform(matches, new Function<NavigationLink, NavigationLink>()
        {
            @Override
            public NavigationLink apply(@Nullable final NavigationLink remoteLink)
            {
                return remoteLink != null ? NavigationLinkBuilder.copyOf(remoteLink).build() : null;
            }
        }));
    }

    @Nonnull
    private List<NavigationLink> mergeNavigationLinks(@Nonnull final Iterable<NavigationLink> localNavigationLinks,
                                                      @Nonnull final Iterable<NavigationLink> remoteNavigationLinks)
    {
        final Iterable<NavigationLink> navigationLinks = Iterables.concat(localNavigationLinks, remoteNavigationLinks);
        final Iterable<NavigationLink> processedNavigationLinks = transform(navigationLinks, MaskBitbucketNavigationLinkTransformer.INSTANCE);
        final List<NavigationLink> result = Lists.newArrayList(processedNavigationLinks);
        Collections.sort(result, NavigationLinkComparator.INSTANCE);
        return result;
    }
}
