package com.atlassian.plugins.navlink.consumer.menu.services;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.TypeNotInstalledException;
import com.atlassian.failurecache.CacheLoader;
import com.atlassian.failurecache.ExpiringValue;
import com.atlassian.plugins.navlink.consumer.CachingApplicationLinkService;
import com.atlassian.plugins.navlink.consumer.menu.client.capabilities.CapabilitiesClient;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;

import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkNotNull;

public class CapabilitiesCacheLoader implements CacheLoader<ApplicationId, RemoteApplicationWithCapabilities>
{
    private final CachingApplicationLinkService applicationLinkService;
    private final CapabilitiesClient capabilitiesClient;

    public CapabilitiesCacheLoader(final CachingApplicationLinkService applicationLinkService, final CapabilitiesClient capabilitiesClient)
    {
        this.applicationLinkService = applicationLinkService;
        this.capabilitiesClient = capabilitiesClient;
    }

    @Override
    public ImmutableSet<ApplicationId> getKeys()
    {
        final Iterable<ApplicationLink> allApplicationLinks = applicationLinkService.getApplicationLinks();
        final Iterable<ApplicationLink> goodApplicationLinks = Iterables.filter(allApplicationLinks, Predicates.not(IgnoreRemotePluginApplicationLinkPredicate.INSTANCE));
        final Iterable<ApplicationId> goodApplicationIds = Iterables.transform(goodApplicationLinks, toApplicationId());
        return ImmutableSet.copyOf(goodApplicationIds);
    }

    @Override
    public ListenableFuture<ExpiringValue<RemoteApplicationWithCapabilities>> loadValue(final ApplicationId applicationId)
    {
        try
        {
            final ApplicationLink applicationLink = getApplicationLink(applicationId);
            return capabilitiesClient.getCapabilities(applicationLink);
        }
        catch (RuntimeException e)
        {
            return Futures.immediateFailedFuture(e);
        }
        catch (TypeNotInstalledException e)
        {
            return Futures.immediateFailedFuture(e);
        }
    }

    private ApplicationLink getApplicationLink(final ApplicationId applicationId) throws TypeNotInstalledException
    {
        return checkNotNull(applicationLinkService.getApplicationLink(applicationId), "Application link with application id '" + applicationId + "' is not existing.");
    }

    private Function<ApplicationLink, ApplicationId> toApplicationId()
    {
        return new Function<ApplicationLink, ApplicationId>()
        {
            @Override
            public ApplicationId apply(@Nullable ApplicationLink from)
            {
                return from != null ? from.getId() : null;
            }
        };
    }
}