package com.atlassian.plugins.navlink.consumer.menu.client.navigation;

import com.atlassian.failurecache.ExpiringValue;
import com.atlassian.plugins.navlink.consumer.http.HttpRequestFactory;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.atlassian.plugins.navlink.producer.navigation.ApplicationNavigationLinks;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import java.util.Locale;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;


/**
 * Returns the {@link NavigationLink}s that an application provides.
 *
 * @since 1.0
 */
public class RestNavigationClient implements NavigationClient, DisposableBean
{
    private static final Logger logger = LoggerFactory.getLogger(RestNavigationClient.class);
    private final HttpRequestFactory httpRequestFactory;
    private final ListeningExecutorService executor;

    public RestNavigationClient(final HttpRequestFactory httpRequestFactory)
    {
        this(httpRequestFactory, MoreExecutors.listeningDecorator(Executors.newSingleThreadExecutor()));
    }

    @VisibleForTesting
    RestNavigationClient(final HttpRequestFactory httpRequestFactory, final ListeningExecutorService executor)
    {
        this.httpRequestFactory = httpRequestFactory;
        this.executor = executor;
    }

    @Override
    public ListenableFuture<ExpiringValue<ApplicationNavigationLinks>> getNavigationLinks(final RemoteApplicationWithCapabilities application, final Locale locale)
    {
        final String requestUrl = NavigationRestResourceUrlFactory.createRequestUrl(application, locale);
        if (requestUrl == null)
        {
            logger.debug("Remote application with link id {} doesn't support the navigation capability. Skipping ...", application.getApplicationLinkId());
            return Futures.immediateFuture(ExpiringValue.<ApplicationNavigationLinks>expiredNullValue());
        }
        else
        {
            logger.debug("Scheduling request for navigation links from '{}' (application link id {})", requestUrl, application.getApplicationLinkId());
            return scheduleRequest(requestUrl, new NavigationLinkResponseHandler(application, locale));
        }
    }

    @Override
    public void destroy() throws Exception
    {
        executor.shutdownNow();
    }

    private ListenableFuture<ExpiringValue<ApplicationNavigationLinks>> scheduleRequest(final String requestUrl, final NavigationLinkResponseHandler responseHandler)
    {
        return executor.submit(new Callable<ExpiringValue<ApplicationNavigationLinks>>()
        {
            @Override
            public ExpiringValue<ApplicationNavigationLinks> call() throws Exception
            {
                final long before = System.currentTimeMillis();
                try
                {
                    return submitRequest(requestUrl, responseHandler);
                }
                finally
                {
                    logger.debug("Navigation links from '{}' have been fetched in {} ms.", requestUrl, System.currentTimeMillis() - before);

                }
            }
        });
    }

    @VisibleForTesting
    ExpiringValue<ApplicationNavigationLinks> submitRequest(final String requestUrl, final NavigationLinkResponseHandler responseHandler)
    {
        try
        {
            return httpRequestFactory.executeGetRequest(requestUrl, responseHandler);
        }
        catch (Exception e)
        {
            return handleException(requestUrl, e);
        }
    }

    private ExpiringValue<ApplicationNavigationLinks> handleException(final String requestUrl, final Exception e)
    {
        logger.info("Failed to request navigation links from '{}': {}", requestUrl, e.getMessage());
        logger.debug("Stacktrace: ", e);
        return ExpiringValue.expiredNullValue();
    }

}
