package com.atlassian.plugins.navlink.consumer.menu.client.capabilities;

import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.failurecache.ExpiringValue;
import com.atlassian.plugins.navlink.consumer.http.HttpRequestFactory;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import java.util.concurrent.Callable;

import static com.atlassian.plugins.navlink.consumer.menu.client.capabilities.CapabilitiesRestResourceUrlFactory.createRequestUrl;
import static com.atlassian.util.concurrent.ThreadFactories.namedThreadFactory;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

public class RestCapabilitiesClient implements CapabilitiesClient, DisposableBean
{
    private static final Logger logger = LoggerFactory.getLogger(RestCapabilitiesClient.class);
    private final HttpRequestFactory requestFactory;
    private final ListeningExecutorService executor;

    public RestCapabilitiesClient(final HttpRequestFactory requestFactory)
    {
        this(requestFactory, MoreExecutors.listeningDecorator(newSingleThreadExecutor(namedThreadFactory("NavLink RestCapabilitiesClient"))));
    }

    @VisibleForTesting
    RestCapabilitiesClient(final HttpRequestFactory requestFactory, final ListeningExecutorService executor)
    {
        this.requestFactory = requestFactory;
        this.executor = executor;
    }

    @Override
    public ListenableFuture<ExpiringValue<RemoteApplicationWithCapabilities>> getCapabilities(final ApplicationLink applicationLink)
    {
        final String requestUrl = createRequestUrl(applicationLink);
        logger.debug("Scheduling request for capabilities from '{}' (application link id: {})", requestUrl, applicationLink.getId());
        return scheduleRequest(requestUrl, new CapabilitiesResponseHandler(applicationLink));
    }

    @Override
    public void destroy() throws Exception
    {
        executor.shutdownNow();
    }

    private ListenableFuture<ExpiringValue<RemoteApplicationWithCapabilities>> scheduleRequest(final String requestUrl, final CapabilitiesResponseHandler responseHandler)
    {
        return executor.submit(new Callable<ExpiringValue<RemoteApplicationWithCapabilities>>()
        {
            @Override
            public ExpiringValue<RemoteApplicationWithCapabilities> call() throws Exception
            {
                final long before = System.currentTimeMillis();
                try
                {
                    return submitRequest(requestUrl, responseHandler);
                }
                finally
                {
                    logger.debug("Capabilities from '{}' have been fetched in {} ms", requestUrl, System.currentTimeMillis() - before);
                }
            }
        });
    }

    @VisibleForTesting
    ExpiringValue<RemoteApplicationWithCapabilities> submitRequest(final String requestUrl, final CapabilitiesResponseHandler responseHandler)
    {
        try
        {
            return requestFactory.executeGetRequest(requestUrl, responseHandler);
        }
        catch (Exception e)
        {
            return handleException(requestUrl, e);
        }
    }

    private ExpiringValue<RemoteApplicationWithCapabilities> handleException(final String requestUrl, final Exception e)
    {
        logger.info("Failed to request capabilities from '{}': {}", requestUrl, e.getMessage());
        logger.debug("Stacktrace: ", e);
        return ExpiringValue.expiredNullValue();
    }
}
