package com.atlassian.plugins.navlink.consumer.menu.client.capabilities;

import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilitiesBuilder;
import com.atlassian.plugins.navlink.util.date.UniversalDateFormatter;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;
import org.apache.http.impl.client.BasicResponseHandler;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.joda.time.DateTime;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.isNullOrEmpty;

public class CapabilitiesResponseHandler implements ResponseHandler<RemoteApplicationWithCapabilities>
{
    private static final String CAPABILITIES = "capabilities";

    protected final ApplicationLink applicationLink;
    protected final ResponseHandler<String> basicHandler = new BasicResponseHandler();

    public CapabilitiesResponseHandler(final ApplicationLink applicationLink)
    {
        this.applicationLink = checkNotNull(applicationLink);
    }

    @Override
    public RemoteApplicationWithCapabilities handleResponse(final HttpResponse response) throws IOException
    {
        final String responseBody = basicHandler.handleResponse(response);
        return StringUtils.isNotBlank(responseBody) ? parseBody(responseBody) : null;
    }

    protected RemoteApplicationWithCapabilities parseBody(final String responseBody) throws IOException
    {
        final ObjectMapper mapper = new ObjectMapper();
        final JsonNode jsonNodes = mapper.readValue(responseBody, JsonNode.class);
        return new RemoteApplicationWithCapabilitiesBuilder()
                .setApplicationLinkId(applicationLink.getId().toString())
                .setSelfUrl(parseSelfUrl(jsonNodes))
                .setType(jsonNodes.path("application").getTextValue())
                .setBuildDate(getBuildDate(jsonNodes))
                .addAllCapabilities(collectCapabilities(jsonNodes))
                .build();
    }

    @Nullable
    private String parseSelfUrl(final JsonNode jsonNodes)
    {
        return jsonNodes.path("links").path("self").getTextValue();
    }

    @Nullable
    private DateTime getBuildDate(final JsonNode jsonNodes)
    {
        final String buildDateString = jsonNodes.path("buildDate").getTextValue();
        if (!isNullOrEmpty(buildDateString))
        {
            return UniversalDateFormatter.parse(buildDateString);
        }
        else
        {
            return null;
        }
    }

    private Map<String, String> collectCapabilities(final JsonNode jsonNodes)
    {
        final ImmutableMap.Builder<String,String> capabilitiesBuilder = ImmutableMap.builder();
        final Iterator<String> capabilities = jsonNodes.path(CAPABILITIES).getFieldNames();
        while (capabilities.hasNext())
        {
            String capabilityName = capabilities.next();
            if (!isNullOrEmpty(capabilityName))
            {
                final String capabilityUrl = jsonNodes.path(CAPABILITIES).path(capabilityName).getTextValue();
                if (!isNullOrEmpty(capabilityUrl))
                {
                    capabilitiesBuilder.put(capabilityName, capabilityUrl);
                }
            }
        }
        return capabilitiesBuilder.build();
    }
}
