package com.atlassian.plugins.navlink.consumer.http;

import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang.time.DateUtils;
import org.apache.http.Header;
import org.apache.http.HeaderElement;
import org.apache.http.HttpResponse;

import javax.annotation.Nullable;
import java.text.ParseException;

/**
 * Find values in the HTTP headers.
 */
public class HeaderSearcher
{
    private static final String[] ACCEPTED_DATETIME_FORMATS = new String[]{DateFormatUtils.SMTP_DATETIME_FORMAT.getPattern()};

    private final HttpResponse httpResponse;

    public HeaderSearcher(final HttpResponse httpResponse)
    {
        this.httpResponse = httpResponse;
    }

    /**
     * Return the value of the first header field with the given name.
     * @param headerFieldName the name of the header field whose value is of interest
     * @return the value of the first header field or <code>null</code> if not found
     */
    @Nullable
    public String findFirstHeaderValue(final String headerFieldName)
    {
        final Header firstHeader = httpResponse.getFirstHeader(headerFieldName);
        return firstHeader != null ? firstHeader.getValue() : null;
    }

    /**
     * Return the value of the first header field with the given name.
     * @param headerFieldName the name of the header field whose value is of interest
     * @return the value of the first header field or <code>null</code> if not found or the conversion to {@link Long}
     * failed
     * @see {@link org.apache.http.Header#getValue()}
     */
    @Nullable
    public Long findFirstHeaderValueAsLong(final String headerFieldName)
    {
        final String headerValue = findFirstHeaderValue(headerFieldName);
        return headerValue != null ? parseLong(headerValue) : null;
    }

    /**
     * Return the valud of the first header field with the given name.
     * @param headerFieldName the name of the header field whose value is of interest
     * @return the value of the first header field or <code>null</code> if not found or it doesn't match the expected
     * date format and thus the conversion failed
     * @see {@link #ACCEPTED_DATETIME_FORMATS}
     */
    @Nullable
    public Long findFirstHeaderValueAsDateInMillis(final String headerFieldName)
    {
        final String headerValue = findFirstHeaderValue(headerFieldName);
        return headerValue != null ? parseDate(headerValue) : null;

    }

    /**
     * Return the header element of the first header field with the given name. A header value can consist of multiple
     * header elements, each separated through comma.
     * @param headerFieldName the name of the header field whose header element is of interest
     * @param headerElementName the name of the element of the header value whose value is of interest
     * @return the value of the header elment of the first header field or <code>null</code> if not found or the
     * conversion to {@link Long} did fail.
     * @see {@link org.apache.http.Header#getElements()}
     * @see {@link org.apache.http.HeaderElement}
     */
    @Nullable
    public Long findFirstHeaderElementAsLong(final String headerFieldName, final String headerElementName)
    {
        final Header firstHeader = httpResponse.getFirstHeader(headerFieldName);
        if (firstHeader != null) {
            final String headerElement = findHeaderElement(firstHeader.getElements(), headerElementName);
            return headerElement != null ? parseLong(headerElement) : null;
        }
        else
        {
            return null;
        }
    }

    /**
     * Return the value of the first header element matching the given name.
     * @param elements the header element in which a certain value should be searched
     * @param headerElementName the name of the header element whose value is of interest
     * @return the value of the header element or <code>null</code> if not found
     */
    @Nullable
    private String findHeaderElement(final HeaderElement[] elements, final String headerElementName)
    {
        for (HeaderElement element : elements)
        {
            if (element.getName().equalsIgnoreCase(headerElementName))
            {
                return element.getValue();
            }
        }
        return null;
    }

    @Nullable
    private Long parseLong(final String longValue)
    {
        try
        {
            return Long.parseLong(longValue);
        }
        catch (NumberFormatException e)
        {
            return null;
        }
    }

    @Nullable
    private Long parseDate(final String dateValue)
    {
        try
        {
            return DateUtils.parseDateStrictly(dateValue, ACCEPTED_DATETIME_FORMATS).getTime();
        }
        catch (ParseException e)
        {
            return null;
        }
    }
}
