package com.atlassian.plugins.navlink.consumer.admin.rest;

import com.atlassian.failurecache.CacheRefreshService;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.websudo.WebSudoRequired;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import java.util.concurrent.CancellationException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import static java.lang.String.format;

/**
 * REST resource to control capabilities consumer configuration.
 *
 * @since 2.1.1
 */
@Path("/admin")
@WebSudoRequired
public class ConsumerAdminResource
{
    private static final Logger logger = LoggerFactory.getLogger(ConsumerAdminResource.class);
    private static final long CACHE_REBUILD_TIMEOUT_IN_SECONDS = 30;

    private final UserManager userManager;
    private final CacheRefreshService cacheRefreshService;

    public ConsumerAdminResource(UserManager userManager, CacheRefreshService cacheRefreshService)
    {
        this.userManager = userManager;
        this.cacheRefreshService = cacheRefreshService;
    }

    @POST
    @Path("/refreshcache")
    @Produces(MediaType.TEXT_PLAIN)
    @Consumes(MediaType.APPLICATION_JSON)
    public Response clearCaches(@Context HttpServletRequest request)
    {
        final String user = userManager.getRemoteUsername(request);
        if (!userManager.isAdmin(user))
        {
            return Response.status(Response.Status.FORBIDDEN).entity(format("User %s does not have admin permission.", user)).build();
        }
        return rebuildCaches();
    }

    private Response rebuildCaches()
    {
        final long before = System.currentTimeMillis();
        final Future<?> rebuildCachesFuture = cacheRefreshService.refreshAll(false);
        try
        {
            rebuildCachesFuture.get(CACHE_REBUILD_TIMEOUT_IN_SECONDS, TimeUnit.SECONDS);
            logger.debug("Caches have been rebuild in {} ms.", System.currentTimeMillis() - before);
            return Response.ok().entity(String.format("Caches have been rebuilt in %d ms.", System.currentTimeMillis() - before)).build();
        }
        catch (InterruptedException e)
        {
            logger.debug("Interrupted while invalidating the caches", e);
            return Response.serverError().entity("Interrupted while rebuilding the caches.").build();
        }
        catch (CancellationException e)
        {
            logger.debug("Cache rebuild has been cancelled", e);
            return Response.ok().entity("Cache rebuild has been cancelled.").build();
        }
        catch (TimeoutException e)
        {
            logger.debug("Timeout exceeded while clearing the caches", e);
            return Response.ok().entity("Timeout exceeded while waiting for cache rebuild. Not all caches may have been rebuild yet but it is still ongoing.").build();
        }
        catch (Exception e)
        {
            logger.debug("Exception occurred while rebuilding the caches", e);
            return Response.serverError().entity("Exception occurred while rebuilding the caches: " + e.getMessage()).build();
        }
    }

}
