package com.atlassian.plugins.custom_apps.rest;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.xml.bind.annotation.XmlRootElement;

import com.atlassian.plugins.custom_apps.api.CustomApp;
import com.atlassian.plugins.custom_apps.api.CustomAppNotFoundException;
import com.atlassian.plugins.custom_apps.api.CustomAppService;
import com.atlassian.plugins.custom_apps.api.CustomAppsValidationException;
import com.atlassian.plugins.custom_apps.rest.data.CustomAppData;
import com.atlassian.plugins.custom_apps.rest.data.MoveBean;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.websudo.WebSudoRequired;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import static com.atlassian.plugins.navlink.util.CacheControlFactory.withNoCache;

@Produces ("application/json")
@Consumes ("application/json")
@Path ("/customapps")
@WebSudoRequired
public class CustomAppsRestResource
{

    private final CustomAppService customAppService;
    private final UserManager userManager;

    public CustomAppsRestResource(final CustomAppService customAppService, final UserManager userManager)
    {
        this.customAppService = customAppService;
        this.userManager = userManager;
    }

    @Path ("list")
    @GET
    public Response list()
    {
        Iterable<CustomAppData> l = Iterables.transform(customAppService.getLocalCustomAppsAndRemoteLinks(), converter());
        return Response.ok(l).cacheControl(withNoCache()).build();
    }

    private Function<CustomApp, CustomAppData> converter()
    {
        return new Function<CustomApp, CustomAppData>()
        {
            @Override
            public CustomAppData apply(@Nullable CustomApp c)
            {
                return new CustomAppData(c.getId(), c.getDisplayName(), c.getUrl(), c.getSourceApplicationType(), c.getHide(), c.getEditable(), c.getAllowedGroups(), c.getSourceApplicationUrl(), c.getSourceApplicationName(), c.isSelf());
            }
        };
    }

    @Path ("{id}")
    @GET
    public Response get(@PathParam ("id") String id, @Context HttpServletRequest request)
    {
        try
        {
            checkAdminPermission(request);
            return Response.ok(converter().apply(customAppService.get(id))).build();
        }
        catch (CustomAppNotFoundException e)
        {
            return Response.status(Response.Status.NOT_FOUND).build();
        }
        catch (PermissionDeniedException e)
        {
            return handleNoPermission();
        }
    }

    /**
     * @param q String prefix to search for
     * @param pageLimit maximum number of results to return
     * @param page page to load - this is 1 based
     * @param request current HTTP request
     * @return a 'Groups' object as JSON, or a 401 response if the remote user is not an Admin.
     */
    @Path ("groups")
    @GET
    public Response get(@QueryParam ("q") String q, @QueryParam ("page_limit") int pageLimit, @QueryParam ("page") int page, @Context HttpServletRequest request)
    {
        try
        {

            checkAdminPermission(request);
            Iterable<String> groups = userManager.findGroupNamesByPrefix(q, (page - 1) * pageLimit, pageLimit + 1);
            Groups groupResponse = new Groups();
            groupResponse.names = new ArrayList<String>();
            Iterator<String> i = groups.iterator();
            for (int index = 0; index < pageLimit && i.hasNext(); ++index)
            {
                groupResponse.names.add(i.next());
            }
            groupResponse.more = i.hasNext();
            return Response.ok(groupResponse).cacheControl(withNoCache()).build();
        }
        catch (PermissionDeniedException e)
        {
            return handleNoPermission();
        }
    }

    @XmlRootElement
    private static class Groups
    {
        private Groups(List<String> names, boolean more)
        {
            this.names = names;
            this.more = more;
        }

        private Groups()
        {
        }

        public List<String> names;
        public boolean more;
    }


    private Response handleNoPermission()
    {
        return Response.status(Response.Status.UNAUTHORIZED).build();
    }

    private void checkAdminPermission(HttpServletRequest request) throws PermissionDeniedException
    {
        if (!userManager.isAdmin(userManager.getRemoteUsername(request)))
        {
            throw new PermissionDeniedException();
        }
    }

    @Path ("{id}")
    @DELETE
    public Response delete(@PathParam ("id") String id, @Context HttpServletRequest request)
    {
        try
        {
            checkAdminPermission(request);
            customAppService.delete(id);
        }
        catch (CustomAppNotFoundException e)
        {
            // ROTP-361: AUI's RESTful Table expects a JSON response in all error cases. However,
            // the underlying application may use its own error page if the response has no body.
            return Response.status(Response.Status.NOT_FOUND).entity(Collections.EMPTY_MAP).build();
        }
        catch (PermissionDeniedException e)
        {
            return handleNoPermission();
        }
        return Response.ok().cacheControl(withNoCache()).build();
    }

    @POST
    public Response create(CustomAppData data, @Context HttpServletRequest request)
    {
        try
        {
            checkAdminPermission(request);
            return Response.ok(converter().apply(customAppService.create(data.displayName, data.url, null, data.hide == null ? false : data.hide, data.allowedGroups))).cacheControl(withNoCache()).build();
        }
        catch (CustomAppsValidationException e)
        {
            return validationErrorResponse(e);
        }
        catch (PermissionDeniedException e)
        {
            return handleNoPermission();
        }
    }

    private Response validationErrorResponse(CustomAppsValidationException e)
    {
        return Response
                .status(Response.Status.BAD_REQUEST)
                .entity("{\"errors\": {\"" + e.getField() + "\": \"" + e.getValidationError() + "\"}}")
                .build();
    }

    @Path ("{id}")
    @PUT
    public Response update(@PathParam ("id") String id, CustomAppData data, @Context HttpServletRequest request)
    {
        try
        {
            checkAdminPermission(request);
            CustomApp c = customAppService.get(id);
            return Response.ok(
                    converter().apply(
                            customAppService.update(
                                    id,
                                    data.displayName == null ? c.getDisplayName() : data.displayName,
                                    data.url == null ? c.getUrl() : data.url,
                                    data.hide == null ? c.getHide() : data.hide,
                                    data.allowedGroups == null ? c.getAllowedGroups() : data.allowedGroups
                            )
                    )
            ).build();
        }
        catch (CustomAppNotFoundException e)
        {
            return Response.status(Response.Status.NOT_FOUND).build();
        }
        catch (CustomAppsValidationException e)
        {
            return validationErrorResponse(e);
        }
        catch (PermissionDeniedException e)
        {
            return handleNoPermission();
        }
    }

    /**
     * Move the custom app at {id} as specified by the MoveBean in the request
     */
    @POST
    @Path ("{id}/move")
    public Response movePosition(
            @PathParam ("id") final Integer id,
            @Context HttpServletRequest request,
            MoveBean bean)
    {
        try
        {
            checkAdminPermission(request);
            if (bean.after != null)
            {
                int idToMoveAfter = extractIdFromLink(bean.after.getPath());
                customAppService.moveAfter(id, idToMoveAfter);
            }
            else
            {
                switch (bean.position)
                {
                    case Earlier:
                    case Later:
                    case Last:
                        throw new IllegalArgumentException("Unexpected position '" + bean.position + "'");
                    case First:
                        customAppService.moveToStart(id);
                }
            }
            return Response.ok().cacheControl(withNoCache()).build();
        }
        catch (PermissionDeniedException e)
        {
            return handleNoPermission();
        }
        catch (CustomAppNotFoundException e)
        {
            return Response.status(Response.Status.NOT_FOUND).build();
        }

    }

    private int extractIdFromLink(String path)
    {
        String idString = path.substring(path.lastIndexOf('/') + 1);
        try
        {
            return Integer.parseInt(idString);
        }
        catch (NumberFormatException e)
        {
            throw new IllegalArgumentException("Failed to parse id from path '" + path + "'");
        }
    }

    private class PermissionDeniedException extends Exception
    {
    }
}
